#ifndef MDAGraph_Header
#define MDAGraph_Header

#include "ModelGeneric.h"

namespace Aztec {
  
  class MDAGraph;
  class MDAGraphDownstreamIterator;
  class MDAGraphUpstreamIterator;
}

#include "MDAGNode.h"

#include <MBaseUndoNode.h>

#include <set>
#include <stack>

namespace Aztec {

  /** 
   * This is a DAG (Directed A-Cyclic Graph) Node. It is an object which 
   * contains a number of inputs and a number of outputs. When created, 
   * it places a pointer of itself with.
   */
  class MGENEXPORT MDAGraph : public MUndoableObject {
  public:
    // Construction/Destruction
    MDAGraph();
    virtual ~MDAGraph();
    
    /**
     * This adds the node to the graph.
     */
    void addNode(MDAGNode *node);

    /**
     * This removes the node from the graph. It will
     * disconnect the node from any inputs outputs first.
     */
    void removeNode(MDAGNode *node);

    /**
     * This flags the given node, and all nodes that it outputs to
     * to have the given flag.
     */
    static void flagOutputs(MDAGNode *node, AztecFlags flag);

    /**
     * This checks to see if the given node eventually
     * outputs to the second node. This is used
     * for checking to see if we will be creating a cyclic graph,
     * which is not good.
     */
    static bool doesNodeOutputTo(MDAGNode *from, MDAGNode *to);
    
    // MUndoableObject methods
    void finishWithUndoNode();

  protected:
    typedef std::set<MDAGNode*> NodeSet;
    NodeSet allNodes;

    class Undo : public MBaseUndoNode {
    public:
      Undo(MDAGraph *graph);

      MUndoableObject* getObject();
      void undo();
      void redo();

    private:
      NodeSet stored;
      MDAGraph *object;
    };

    MRefCountedPtr<Undo> undo;

    void ensureUndo();

    friend class Undo;
  };

  class MGENEXPORT MDAGraphDownstreamIterator {
  public:
    MDAGraphDownstreamIterator();
    MDAGraphDownstreamIterator(MDAGNode *node);
    MDAGraphDownstreamIterator(MDAGraph *graph, MDAGNode *node);
    MDAGraphDownstreamIterator(const MDAGraphDownstreamIterator &src);
    ~MDAGraphDownstreamIterator();

    MDAGNode* operator*();
    MDAGNode* operator->();

    // prefix increment
    MDAGraphDownstreamIterator& operator++();

    // postfix increment
    MDAGraphDownstreamIterator operator++(int);

    // has this iterator hit the end yet?
    bool atEnd() {
      return curNode == NULL;
    }

  protected:
    MDAGraph *graph;
    MDAGNode *curNode;

    std::set<MDAGNode*> visited;
    std::stack<MDAGNode*> tovisit;

    bool haveVisited(MDAGNode *node);

    // this adds the outputs of the current
    // node to visit, if we haven't already visited them
    void addOutputs(MDAGNode *node);

  };

  class MGENEXPORT MDAGraphUpstreamIterator {
  public:
    MDAGraphUpstreamIterator();
    MDAGraphUpstreamIterator(MDAGNode *node);
    MDAGraphUpstreamIterator(MDAGNode *node, const std::vector<MDAGNode*> &ignoreList);
    MDAGraphUpstreamIterator(MDAGraph *graph, MDAGNode *node);
    MDAGraphUpstreamIterator(const MDAGraphUpstreamIterator &src);
    ~MDAGraphUpstreamIterator();

    MDAGNode* operator*();
    MDAGNode* operator->();

    // prefix increment
    MDAGraphUpstreamIterator& operator++();

    // postfix increment
    MDAGraphUpstreamIterator operator++(int);

    // has this iterator hit the end yet?
    bool atEnd() {
      return curNode == NULL;
    }

  protected:
    MDAGraph *graph;
    MDAGNode *curNode;

    std::set<MDAGNode*> visited;
    std::stack<MDAGNode*> tovisit;

    bool haveVisited(MDAGNode *node);

    // this adds the outputs of the current
    // node to visit, if we haven't already visited them
    void addInputs(MDAGNode *node);

  };
  

}


#endif
