#ifndef MBinaryFileStream_Header
#define MBinaryFileStream_Header

#include "ModelGeneric.h"

namespace Aztec {
  class MBinaryFileReader;
  class MBinaryFileWriter;

  typedef MRefCountedPtr<MBinaryFileReader> MBinaryFileReaderPtr;
  typedef MRefCountedPtr<MBinaryFileWriter> MBinaryFileWriterPtr;

}

#include "MStreams.h"

#include <stdio.h>

namespace Aztec {

  class MGENEXPORT MBinaryFileReader : public MInputStream {
  public:
    /**
     * Default constructor, with no stream being opened.
     */
    MBinaryFileReader();
    /**
     * Constructor with a given filename, and attempts to open
     * the given file for reading.
     */
    MBinaryFileReader(const MStr &filename);

    /**
     * destructor
     */
    virtual ~MBinaryFileReader();

    /**
     * opens the stream to read from the given filename
     */
    bool open(const MStr &filename);
    /**
     * Closes the current stream.
     */
    void close();
    /**
     * Returns true if the stream is currently open
     */
    bool isOpen();
    /**
     * Returns true if we are currently at the end of the stream.
     */
    bool isAtEOF();

    // methods from MInputStream
    char readChar();
    unsigned char readUnsignedChar();
    short readShort();
    unsigned short readUnsignedShort();
    int readInt();
    unsigned int readUnsignedInt();
    float readFloat();
    double readDouble();
    bool readBoolean();
    MStr readString();
    int readBuffer(void *buffer, int bufSize);

  protected:
    /**
     * Internal file handle to use
     */
    FILE *m_FileHandle;

  };

  /**
   * This is a class that writes text files. 
   */
  class MGENEXPORT MBinaryFileWriter : public MOutputStream {
  public:
    /**
     * Default constructor, with no stream being opened.
     */
    MBinaryFileWriter();
    /**
     * Constructor with a given filename, and attempts to open
     * the given file for writing.
     */
    MBinaryFileWriter(const MStr &filename, bool append = false);
    /**
     * destructor
     */
    virtual ~MBinaryFileWriter();

    /**
     * opens the stream to write to from the given filename.
     * It will append the file if required, otherwise it
     * will create/overwrite it.
     *
     * @param filename The filename of th file to open
     * @param append This appends the file if this is 
     *               true, false otherwise
     * @return true if the file was sucessfully opened.
     */
    bool open(const MStr &filename, bool append = false);
    /**
     * Clsoes the current stream.
     */
    void close();
    /**
     * Returns true if the stream is currently open
     */
    bool isOpen();
    /**
     * Returns true if we are currently at the end of the stream.
     */
    bool isAtEOF();

    // methods from MOutputStream
    void writeChar(char value);
    void writeUnsignedChar(unsigned char value);
    void writeShort(short value);
    void writeUnsignedShort(unsigned short value);
    void writeInt(int value);
    void writeUnsignedInt(unsigned int value);
    void writeFloat(float value);
    void writeDouble(double value);
    void writeBoolean(bool value);
    void writeString(const MStr &value);
    void writeString(const char *value);
    void writeBuffer(const void*buffer, int bufSize);


  protected:
    /**
     * Internal file handle
     */
    FILE *m_FileHandle;

  };

}

#endif
