#ifndef MTreeControl_Header
#define MTreeControl_Header

#include "MRefCountedPtr.h"

namespace Aztec {

  class IMTreeControl;
  class MTreeControl;
  class MTreeItem;

  typedef MRefCountedPtr<IMTreeControl> IMTreeControlPtr;
  typedef MRefCountedPtr<MTreeControl> MTreeControlPtr;
  typedef MRefCountedPtr<MTreeItem> MTreeItemPtr;


}

#include <gui/MComponent.h>
#include <string>

namespace Aztec {

  /**
   * The base class for an item in the MTreeControl class.
   */
  class AZTECGUI_EXPORT IMTreeItem : public virtual MRefCountedObject {
  public:
    /**
     * This gets the text that is displayed for this item.
     */
    virtual std::string getText() const = 0;

    /**
     * Compres this tree item with the given tree item.
     *
     * @return -1 if this < rhs, 0 if this == rhs, and 1 if this > rhs.
     */
    virtual int compare(const MTreeItemPtr &rhs) = 0;

    virtual MTreeItemPtr getParent() const = 0;

    virtual bool getSelected() const = 0;
  };


  /**
   * This is a simple tree control that displayed text items in a tree like 
   * structure. It allows the user to select items, and view  only parts of 
   * the tree at a time by collapsing/expanding out branches of the tree.
   */
  class AZTECGUI_EXPORT IMTreeControl : public virtual IMComponent {
  public:
    typedef IMTreeControlPtr Ptr;

    /**
     * This selects the given item, and will cause onItemSelected() to be called.
     * @param item The item to select.
     */
    virtual void selectItem(const MTreeItemPtr &item, bool selected) = 0;

    /** 
     * This clears out the tree so that there are no nodes in the tree.
     */
    virtual void clear() = 0;

    /**
     * This adds an item to tree, under the given parent. If the parent item is NULL, then the given text is added at the root level of the tree.
     * 
     * @parem text The text to add as an item.
     * @param parent The item under which we are adding the new item.
     * @return the new tree item that was created. NULL if the item could not be created.
     */
    virtual MTreeItemPtr addItem(const std::string &text, const MTreeItemPtr &parent) = 0;

    /**
     * This makes it so the given tree item is expanded out or collapsed.
     *
     * @param item The item to expand/collapse.
     * @expand true if we want the tree item to expand, false if we wan't it to collapse.
     */
    virtual void expandItem(const MTreeItemPtr &item, bool expand) = 0;

    /**
     * This is called when the a different item is selected in the tree view. The item that has recieved selection is passed in as the argument as the method.
     *
     * @param item The item that has just been selected, NULL if nothing is selected.
     * @return true if the event has been handled.
     */
    virtual bool onItemSelected(const MTreeItemPtr &item) = 0;


  };

}


/**
 * The following header is included via the AZTEC_GUI
 * variable that should be defined for each platform
 * and used to specify an include directory
 * (ie. ../AztecGUICommon/include/gui/$(AZTEC_GUI)
 * when compiling this lib
 */
#include <MTreeControlImpl.h>



#endif

