#include <Aztec3DPCH.h>
#include <views/AztecView.h>

// Aztec2 includes
#include <views/AztecViewManager.h>

// AztecGUICommon includes
#include <config/UIConfig.h>
#include <config/ColourConstants.h>
#include <gui/MBorderLayout.h>
#include <gui/MMenuBar.h>
#include <gui/MMenuItem.h>
#include <gui/MMenu.h>
#include <gui/util/MMenuFactory.h>


#ifndef _WIN32
#include <qpainter.h>
#include <iostream>
#endif


namespace AztecGUI {

  AztecViewPtr AztecView::getViewForComponent(const Aztec::MComponentPtr &component) {
    Aztec::MComponentPtr comp = component;

    while (comp != NULL) {
      AztecViewPtr view = AZTEC_CAST(AztecView, comp);

      if (view != NULL) {
        return view;
      } else {
        comp = comp->getParent();
      }
    }

    // return null if we couldn't find anything.
    return NULL;
  }


  AztecView::AztecView(const std::string &name) {
    viewName = name;
    AztecViewManager::registerView(this);
  }

  AztecView::~AztecView() {
    AztecViewManager::unregisterView(this);
  }
  
  std::string AztecView::getName() {
    return viewName;
  }

  void AztecView::setName(const std::string &name) {
    viewName = name;
  }

#ifdef _WIN32
  static DWORD colourToDWORD(const Aztec::MVector3 &col) {
    int r, g, b;

    r = col.x * 255.0;
    g = col.y * 255.0;
    b = col.z * 255.0;

    if (r < 0) r = 0;
    if (r > 255) r = 255;
    if (g < 0) g = 0;
    if (g > 255) g = 255;
    if (b < 0) b = 0;
    if (b > 255) b = 255;

    return RGB(r,g,b);
  }

  static Aztec::MVector3 DWORDToColour(DWORD col) {
    Aztec::MVector3 result;

    result.x = GetRValue(col) / 255.0;
    result.y = GetGValue(col) / 255.0;
    result.z = GetBValue(col) / 255.0;

    return result;
  }
#else
#warning TODO: Cross platform color macros
#endif

  void AztecView::drawBorder(const Aztec::MRect2D &r, int width) {
#ifdef _WIN32
    RECT rect;
    HBRUSH b;

    Aztec::MVector3 flat = DWORDToColour(::GetSysColor(COLOR_3DFACE));
    Aztec::MVector3 border;
    float alpha = 1.0;
    if (AztecViewManager::getCurrentView() == this) {
      Aztec::MVector4 tempBorder = UIConfig::getColourWithAlpha(Colours::VIEW_BORDER_CURRENT);
      border = tempBorder;
      alpha = tempBorder.w;
    } else {
      Aztec::MVector4 tempBorder = UIConfig::getColourWithAlpha(Colours::VIEW_BORDER);
      border = tempBorder;
      alpha = tempBorder.w;
    }

    border = alpha * border + (1.0 - alpha) * flat;

    b = ::CreateSolidBrush(colourToDWORD(border));

    rect.left = r.getLeft();
    rect.top = r.getTop();
    rect.right = r.getRight();
    rect.bottom = r.getTop() + width;
    ::FillRect(paintDC, &rect, b);

    rect.left = r.getLeft();
    rect.top = r.getBottom() - width;
    rect.right = r.getRight();
    rect.bottom = r.getBottom();
    ::FillRect(paintDC, &rect, b);

    rect.left = r.getLeft();
    rect.top = r.getTop();
    rect.right = r.getLeft() + width;
    rect.bottom = r.getBottom();
    ::FillRect(paintDC, &rect, b);

    rect.left = r.getRight() - width;
    rect.top = r.getTop();
    rect.right = r.getRight();
    rect.bottom = r.getBottom();
    ::FillRect(paintDC, &rect, b);

    ::DeleteObject(b);

#else

    // NOTE: This is not the final code
    //       Just testing things to see what's nice..

    QColorGroup cg = m_Handle->colorGroup();
    QRect rc = m_Handle->rect();
    QPainter p;

    // Create the brushes, pens, etc.
    QBrush emptybrush(Qt::NoBrush);
    QBrush fillerbrush(cg.button());
    QPen outline(cg.dark(), 1);

    // Start painting our internal Qt widget
    p.begin(m_Handle);
      p.setBrush(fillerbrush);
      p.drawRect(rc);

      p.setBrush(emptybrush);
      p.setPen(outline);
      p.drawRect(rc);
    p.end();

#endif
  }

  MToolTypePtr AztecView::getSelectTool() {
    return NULL;
  }

  void AztecView::handleToolResult(int toolResult) {
    if (toolResult == TOOLRESULT_DRAWCURRENT) {
      drawView();
    } else if (toolResult == TOOLRESULT_DRAWALL) {
      // update the displays.
      AztecViewManager::redrawAllViews();
    }
  }

  Aztec::MMenuBarPtr AztecView::getMenuBar() const {
    return viewMenuBar;
  }

  void AztecView::displayContextMenu(const Aztec::MMouseEvent &event) {
    Aztec::MMenuPtr menu = new Aztec::MMenu();

    constructContextMenu(menu, event);

    menu->popupMenu(event.getComponent(), event.getPos());
  }


  void AztecView::onCreate() {
    MContainer::onCreate();

    // set up our layout manager
    setLayoutManager(new Aztec::MBorderLayout(2,2));

    // add in a menu bar
    {
      viewMenuBar = new Aztec::MMenuBar();
      //viewMenuBar->addItem(item);

      Aztec::MMenuFactory::loadMenu(viewMenuBar, "menus/aztecView.menu");

      addComponent(viewMenuBar, Aztec::MBorderLayout::NORTH);

      // TODO: On AGUI/Qt, we cannot yet add items to a menubar
      //       that's not added to the borderlayout (because
      //       items are added using the internal Qt widget,
      //       if we do it before this widget doesn't exist yet),
      //       fix this (rve).
//      viewMenuBar->addItem(item);
    }
  }

  bool AztecView::onPaint() {
    drawView();

    drawBorder(getClientSize(), 2);

    return true;
  }

  void AztecView::constructContextMenu(const Aztec::MMenuShellPtr &menu, const Aztec::MMouseEvent &event) {
    Aztec::MMenuFactory::loadMenu(menu, "menus/aztecView.menu");
    return;
  }

  
}

