#include <Aztec3DPCH.h>

// Aztec2 includes
#include <views/Aztec3DView.h>
#include <views/AztecViewManager.h>
#include <controls/Aztec3DSceneCanvas.h>
#include <gui/MBorderLayout.h>
#include <gui/MMenuBar.h>
#include <gui/MMenuItem.h>
#include <gui/MMenu.h>
#include <gui/util/MMenuFactory.h>
#include <tools/MSelectTool.h>
#include <utils/SceneFunctions.h>

// AztecGuiCommon
#include <tools/MToolManager.h>

// AztecLib includes
#include <scripting/MScriptInterp.h>
#include <MUIManager.h>


namespace AztecGUI {

  std::string Aztec3DView::VIEW_GROUP = "3D";

  Aztec3DView::Aztec3DView(const std::string &name, ViewType type,
                Aztec3DSceneCanvas::ShadingModeEnum shading,
                Aztec3DSceneCanvas::TextureModeEnum texturing)
    : AztecGLView(name)
  {
    MToolManager::getInstance()->setDefault(new MSelectToolType(), getViewGroup());

    viewType = type;

    canvas = new Aztec3DSceneCanvas(this);
    canvas->setTextureMode(texturing);
    canvas->setShadingMode(shading);
  }

  Aztec3DView::~Aztec3DView() {
  }

  bool Aztec3DView::deleteSelected() {
    Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.editObjectDelete();");
    return true;
  }

  void Aztec3DView::zoomIn() {
    if (get3DCanvas()->isPerspective()) {
      get3DCanvas()->setDistance(get3DCanvas()->getDistance() / 1.2);
    } else {
      get3DCanvas()->setZoomFactor(get3DCanvas()->getZoomFactor() * 1.2);
    }
  }

  void Aztec3DView::zoomOut() {
    if (get3DCanvas()->isPerspective()) {
      get3DCanvas()->setDistance(get3DCanvas()->getDistance() * 1.2);
    } else {
      get3DCanvas()->setZoomFactor(get3DCanvas()->getZoomFactor() / 1.2);
    }
  }

  void Aztec3DView::zoomToFitAll() {
    zoomToFit(false);
  }

  void Aztec3DView::zoomToFitSelected() {
    zoomToFit(true);
  }

  void Aztec3DView::setViewType(ViewType type, bool changeName) {
    viewType = type;
    if (canvas != NULL) {
      updateCanvasFromViewType(canvas, viewType);
    }
  }

  Aztec3DSceneCanvasPtr Aztec3DView::get3DCanvas() {
    return canvas;
  }

  AztecGLCanvasPtr Aztec3DView::getCanvas() {
    return canvas;
  }

  void Aztec3DView::drawView() {
    // do drawing stuff here.
  }

  AztecViewPtr Aztec3DView::createCopy() {
    Aztec3DViewPtr result = new Aztec3DView(getName());

    // Copy important information about cameras and whatever else here.
    result->viewType = viewType;

    result->canvas->setTextureMode(canvas->getTextureMode());
    result->canvas->setShadingMode(canvas->getShadingMode());
    result->canvas->setBackfaceCulling(canvas->getBackfaceCulling());
    result->canvas->setHeadlight(canvas->getHeadlight());

    return result;
  }

  std::string Aztec3DView::getViewGroup() const {
    return VIEW_GROUP;
  }

  void Aztec3DView::make3DMenu(const Aztec::MMenuShellPtr &menu, bool atStart) {
    
    Aztec::MMenuFactory::loadMenu(viewMenuBar, 0, "menus/aztec3DViewShading.menu");
    Aztec::MMenuFactory::loadMenu(viewMenuBar, 1, "menus/aztec3DViewDisplay.menu");

    addComponent(viewMenuBar, Aztec::MBorderLayout::NORTH);
    
  }
  
  //The Context Sensitive Menu.
  void Aztec3DView::constructContextMenu(const Aztec::MMenuShellPtr &menu, const Aztec::MMouseEvent &event) {
    Aztec::MMenuPtr modeMenu = new Aztec::MMenu();
    Aztec::MMenuPtr selectionMenu = new Aztec::MMenu();
    
    // Now find the select mode and build the context sensitive menu.
    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE)
    {
      Aztec::MMenuFactory::loadMenu(menu, "menus/quickMenuObjectItems.menu");            
    }
    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE)
    {
      Aztec::MMenuFactory::loadMenu(menu, "menus/quickMenuFaceItems.menu");    
    }
    else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE)
    {
      Aztec::MMenuFactory::loadMenu(menu, "menus/quickMenuPointItems.menu");
    }
    else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE)
    {
      
      Aztec::MMenuFactory::loadMenu(menu, "menus/quickMenuEdgeItems.menu");
    }
    Aztec::MMenuFactory::loadMenu(menu, "menus/quickMenu.menu");
    Aztec::MMenuFactory::loadMenu(menu, "menus/aztec3DViewDisplay.menu");
    Aztec::MMenuFactory::loadMenu(menu, "menus/aztec3DViewShading.menu");
    AztecView::constructContextMenu(menu, event);
  }

  void Aztec3DView::onCreate() {
    AztecView::onCreate();

    Aztec::MMenuBarPtr menuBar = getMenuBar();
    make3DMenu(menuBar, true);

    addComponent(canvas, Aztec::MBorderLayout::CENTRE);
    updateCanvasFromViewType(canvas, viewType);
  }

  void Aztec3DView::updateCanvasFromViewType(const Aztec3DSceneCanvasPtr &canvas, ViewType type) {
    switch (type) {
    case Persp :
      canvas->setPerspective(true);
      break;

    case Top :
      canvas->setPerspective(false);
      canvas->setCameraRotation(Aztec::MVector3(0,0,0));
      canvas->setGridBasis(Aztec::MVector3(1,0,0), Aztec::MVector3(0,1,0));
      break;
      
    case Bottom :
      canvas->setPerspective(false);
      canvas->setCameraRotation(Aztec::MVector3(180,0,0));
      canvas->setGridBasis(Aztec::MVector3(1,0,0), Aztec::MVector3(0,1,0));
      break;
      
    case Front :
      canvas->setPerspective(false);
      canvas->setCameraRotation(Aztec::MVector3(-90,0,0));
      canvas->setGridBasis(Aztec::MVector3(1,0,0), Aztec::MVector3(0,0,1));
      break;
      
    case Back :
      canvas->setPerspective(false);
      canvas->setCameraRotation(Aztec::MVector3(-90,0,180));
      canvas->setGridBasis(Aztec::MVector3(1,0,0), Aztec::MVector3(0,0,1));
      break;
      
    case Left :
      canvas->setPerspective(false);
      canvas->setCameraRotation(Aztec::MVector3(-90,0,90));
      canvas->setGridBasis(Aztec::MVector3(0,1,0), Aztec::MVector3(0,0,1));
      break;
      
    case Right :
      canvas->setPerspective(false);
      canvas->setCameraRotation(Aztec::MVector3(-90,0,-90));
      canvas->setGridBasis(Aztec::MVector3(0,1,0), Aztec::MVector3(0,0,1));
      break;
      
    }

  }


  template <class T>
  class CoordinateScanner {
  public:
    CoordinateScanner(bool selectedOnly, 
                      T &grab) 
      : doSelected(selectedOnly), grabber(grab)
    {
    }

    void operator()(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
      // see if we have a scene object
      Aztec::MSceneObjectPtr sceneObj = AZTEC_CAST(Aztec::MSceneObject, node->getObject());

      if (sceneObj != NULL) {
        // see if we have a component object
        Aztec::MComponentisedObjectPtr compObj = sceneObj->getComponentObject();

        // if we have a component object, add in all the points for it, 
        if (compObj != NULL) {

          if (compObj->isInComponentMode()) {
            const Aztec::AztecFlags compMode = Aztec::MUIManager::getComponentMode();
            for (int i = 0; i < compObj->getComponentCount(compMode); ++i) {
              if (!doSelected || compObj->isComponentFlagged(compMode, i)) {
                grabber.addPoint(scene->objectToWorldSpace(node, compObj->getComponentCentre(compMode, i)));
              }
            }
          } else if (compObj->isInComponentMode(Aztec::MComponentisedObject::OBJECT_TYPE)) {
            for (int i = 0; i < compObj->getComponentCount(Aztec::MComponentisedObject::POINT_TYPE); ++i) {
              grabber.addPoint(scene->objectToWorldSpace(node, compObj->getComponentCentre(Aztec::MComponentisedObject::POINT_TYPE, i)));
            }
          }

        } else {
          grabber.addPoint(scene->objectToWorldSpace(node, Aztec::MVector3(0,0,0)));
        }


      }
    }

    bool doSelected;
    T &grabber;
  };

  class Grabber {
  public:
    Grabber(Aztec::MVector3 nu, Aztec::MVector3 nv, Aztec::MVector3 newCentre) 
      : u(nu), v(nv), centre(newCentre)
    {
      count = 0;
      w = u/v;
    }

    static void doCompare(const Aztec::MVector3 &point, Aztec::MVector3 &min, Aztec::MVector3 &max) {
      if (point.x < min.x) min.x = point.x;
      if (point.y < min.y) min.y = point.y;
      if (point.z < min.z) min.z = point.z;
      if (point.x > max.x) max.x = point.x;
      if (point.y > max.y) max.y = point.y;
      if (point.z > max.z) max.z = point.z;
    }

    void addPoint(const Aztec::MVector3 &worldPoint) {
      Aztec::MVector3 centredPoint = worldPoint - centre;
      Aztec::MVector3 point(centredPoint * u, centredPoint * v, centredPoint * w);

      if (count == 0) {
        min = point;
        max = point;
        worldMin = worldPoint;
        worldMax = worldPoint;
      } else {
        doCompare(point, min, max);
        doCompare(worldPoint, worldMin, worldMax);
      }
      ++count;
    }

    Aztec::MVector3 min;
    Aztec::MVector3 max;
    Aztec::MVector3 worldMin;
    Aztec::MVector3 worldMax;
    Aztec::MVector3 u, v, w;
    Aztec::MVector3 centre;
    int count;
  };

  void Aztec3DView::zoomToFit(bool selected) {
    Aztec3DSceneCanvasPtr canvas = get3DCanvas();

    // TODO: make it so it does actual zooming, instead of just moving the
    // focus of the camera to the centre of the selection.
    Aztec::MVector3 centre = Aztec::MScene::getGlobalScene()->getSelectionCentre();
    canvas->setCameraFocus(centre);

    Aztec::MVector3 u, v;
    canvas->getScreenVectors(u, v);
    u.normalize();
    v.normalize();

    std::vector<Aztec::MVector3> points;

    // okay, loop over everything, and whatever is selected, add in to the position list.
    Grabber grabber(u, v, canvas->getCameraFocus());
    CoordinateScanner<Grabber> grabCoordinates(selected, grabber);

    applyToObjects(Aztec::MScene::getGlobalScene(), Aztec::MScene::getGlobalScene()->getObjectList(), selected ? ifSelectedCriteria : allObjectsCriteria, grabCoordinates);

    // if we are zooming to selected objects, and we didn't zoom to anything, just try again but without selection
    if (selected && grabber.count == 0) {
      zoomToFit(false);
      return;
    }

    // okay, now that we have some decent extents, widen them an arbitrary amount, and then widen by a percentage.

    // This allows us to have a nice zoom amount for even miniscule objects, and a consitent border when dealing with large objects.
    grabber.min.x -= 2;
    grabber.min.y -= 2;
    grabber.min.z -= 2;
    grabber.max.x += 2;
    grabber.max.y += 2;
    grabber.max.z += 2;

    Aztec::MVector3 width = grabber.max - grabber.min;
    grabber.min -= width * 0.05;
    grabber.max += width * 0.05;

    // wet the camera so it is looking at the centre.
    canvas->setCameraFocus(0.5 * (grabber.worldMax + grabber.worldMin));

    if (canvas->isPerspective()) {
      // now make our camera fit this dimensions.
      float distX = sqrt((double)2) / 2 * (grabber.max.x - grabber.min.x);
      float distY = sqrt((double)2) / 2 * (grabber.max.y - grabber.min.y);

      // work out the viewing distance based on the field of view of 90 degrees.
      float newDist = (distX > distY) ? distX : distY;
      newDist += 0.5 * (grabber.max.z - grabber.min.z);

      canvas->setDistance(newDist);
    } else {
      float top, bottom, left, right;
      canvas->getGate(left, top, right, bottom);
      float gateWidth = right - left;
      float gateHeight = top - bottom;
      float gateAspect = gateWidth / gateHeight;

      float objWidth = grabber.max.x - grabber.min.x;
      float objHeight = grabber.max.y - grabber.min.y;
      float objAspect = objWidth / objHeight;

      float gateSize = (objAspect > gateAspect) ? gateWidth: gateHeight;
      float objectSize = (objAspect > gateAspect) ? objWidth : objHeight;

      canvas->setZoomFactor(gateSize / objectSize);
    }

  }

}


