#include <Aztec3DPCH.h>

#include <functions/view/ViewFunctions.h>

// Aztec2 includes
#include <functions/file/FileFunctions.h>
#include <views/AztecViewManager.h>
#include <views/ZoomingView.h>
#include <views/ImageView.h>
#include <controls/OptionsWindow.h>
#include <controls/MultiSplitter.h>

// AztecGUI includes
#include <gui/MToolWindow.h>
#include <gui/MBorderLayout.h>
#include <gui/MGridLayout.h>
#include <gui/MSaveFileDialog.h>
#include <gui/MApplication.h>

// AztecLib includes
#include <MScene.h>
#include <MSystemManager.h>
#include <MImages.h>


namespace AztecGUI {

  static int viewZoomIn(const StringVector &args, std::string &result) {
    ZoomingView *zoom = AZTEC_CAST(ZoomingView, AztecViewManager::getCurrentView());
    
    if (zoom != NULL) {
      zoom->zoomIn();
      AztecViewManager::redrawCurrentView();
      return FunctionManager::SUCCEED;
    }
    return FunctionManager::FAIL;
  }

  static int viewZoomOut(const StringVector &args, std::string &result) {
    ZoomingView *zoom = AZTEC_CAST(ZoomingView, AztecViewManager::getCurrentView());
    
    if (zoom != NULL) {
      zoom->zoomOut();
      AztecViewManager::redrawCurrentView();
      return FunctionManager::SUCCEED;
    }
    return FunctionManager::FAIL;
  }

  static int viewZoomToFitSelected(const StringVector &args, std::string &result) {
    ZoomingView *zoom = AZTEC_CAST(ZoomingView, AztecViewManager::getCurrentView());
    
    if (zoom != NULL) {
      zoom->zoomToFitSelected();
      AztecViewManager::redrawCurrentView();
      return FunctionManager::SUCCEED;
    }
    return FunctionManager::FAIL;
  }

  static int viewZoomToFitAll(const StringVector &args, std::string &result) {
    ZoomingView *zoom = AZTEC_CAST(ZoomingView, AztecViewManager::getCurrentView());
    
    if (zoom != NULL) {
      zoom->zoomToFitAll();
      AztecViewManager::redrawCurrentView();
      return FunctionManager::SUCCEED;
    }
    return FunctionManager::FAIL;
  }


  static int viewCreateCopy(const StringVector &args, std::string &result) {
    AztecViewPtr view = AztecViewManager::getCurrentView();

    if (view != NULL) {
      Aztec::MWindowPtr window = new Aztec::MToolWindow(view->getName());
      window->create();
      window->setLayoutManager(new Aztec::MBorderLayout());

      // add in our new view.
      window->addComponent(view->createCopy(), Aztec::MBorderLayout::CENTRE);

      window->setVisible(true);
      return FunctionManager::SUCCEED;
    }

    return FunctionManager::FAIL;
  }

  static int viewChangeCurrent(const StringVector &args, std::string &result) {
    if (args.size() != 1) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: viewChangeCurrent() requies one argument which indicates what view type to change to. For example: Scene.viewChangeCurrent('graph') to change the current view to a graph view.");
      return FunctionManager::FAIL;
    }

    const std::string &viewType = args[0];

    AztecViewPtr oldView = AztecViewManager::getCurrentView();
    if (oldView == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: viewChangeCurrent() requies a current view to work with. This is a bug! please log it at http://sourceforge.net/projects/aztec");
      return FunctionManager::FAIL;
    }

    Aztec::MContainerPtr container = oldView->getParent();

    // we casn't do anything if we are trying to change a view that doesn't exist in a container.
    if (container == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: viewChangeCurrent() : view does not have a parent! This is a bug. please log it at http://sourceforge.net/projects/aztec");
      return FunctionManager::FAIL;
    }

    AztecViewPtr newView = AztecViewManager::createView(viewType);

    if (newView != NULL) {
      Aztec::MConstraint constraint = container->getLayoutManager()->getConstraint(oldView);
      container->removeComponent(oldView);
      container->addComponent(newView, constraint);

      AztecViewManager::setCurrentView(newView);
      return FunctionManager::SUCCEED;
    } else {
      Aztec::MSystemManager::getInstance()->logOutput("Error: viewChangeCurrent() - could not find the view type '%s' to create", viewType.c_str());
      return FunctionManager::FAIL;
    }

  }

  AztecViewPtr viewCreateNew(const std::string &viewType) {
    AztecViewPtr newView = AztecViewManager::createView(viewType);

    if (newView != NULL) {
      // create a new window for this to exist in.
      Aztec::MWindowPtr window = new Aztec::MToolWindow(newView->getName());
      window->create();
      window->setLayoutManager(new Aztec::MBorderLayout());

      // add in our new view.
      window->addComponent(newView, Aztec::MBorderLayout::CENTRE);

      window->setVisible(true);
    }

    return newView;
  }

  static int viewCreateNew(const StringVector &args, std::string &result) {
    if (args.size() != 1) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: viewCreateNew() requies one argument which indicates what view type to change to. For example: Scene.viewChangeCurrent('graph') to create a new graph view.");
      return FunctionManager::FAIL;
    }

    const std::string &viewType = args[0];

    AztecViewPtr newView = viewCreateNew(viewType);

    if (newView != NULL) {
      return FunctionManager::SUCCEED;
    } else {
      Aztec::MSystemManager::getInstance()->logOutput("Error: viewCreateNew() - could not find the view type '%s' to create", viewType.c_str());
      return FunctionManager::FAIL;
    }
  }

  static int showWindow(const StringVector &args, std::string &result) {
    if (args.size() != 1) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: showWindow() - requires one argument indicating which window should be shown.");
      return FunctionManager::FAIL;
    }

    if (args[0] == "configure") {
      OptionsWindow::show();
    }

    return FunctionManager::SUCCEED;
  }

  static void errhandler(const char *str) {
    Aztec::MSystemManager::getInstance()->logOutput(str);
  }

  static Aztec::MImagePtr captureCurrentView() {

    // get the current view.
    Aztec::MImagePtr actualImage;
    AztecViewPtr view = AztecViewManager::getCurrentView();

    if (view == NULL) {
      return false;
    }

    view->refresh();

#ifdef _WIN32
    RECT viewRect;
    ::GetWindowRect(view->getHWnd(), &viewRect);
    int imageX = viewRect.right - viewRect.left;
    int imageY = viewRect.bottom - viewRect.top;

    HDC hdcScreen = CreateDC("DISPLAY", NULL, NULL, NULL); 
    HDC hdcCompatible = CreateCompatibleDC(hdcScreen); 
    
    // Create a compatible bitmap for hdcScreen. 
    int screenX = GetDeviceCaps(hdcScreen, HORZRES);
    int screenY = GetDeviceCaps(hdcScreen, VERTRES);
    
    HBITMAP hbmScreen = CreateCompatibleBitmap(hdcScreen, screenX, screenY); 
    
    if (hbmScreen == 0) { 
      errhandler("hbmScreen"); 
    } else {
    
      // Select the bitmaps into the compatible DC. 
    
      if (!SelectObject(hdcCompatible, hbmScreen)) {
        errhandler("Compatible Bitmap Selection"); 
      } else {
    
        //Copy color data for the entire display into a 
        //bitmap that is selected into a compatible DC. 
    
        if (!BitBlt(hdcCompatible, 0,0, screenX, screenY, 
                    hdcScreen, 0,0, 
                    SRCCOPY)) 
        {
          errhandler("Screen to Compat Blt Failed"); 
        } else {
          // now copy the information out so we can save the image out.
          BITMAPINFOHEADER bInfo;
          bInfo.biSize = sizeof(BITMAPINFOHEADER);
          bInfo.biWidth = screenX;
          bInfo.biHeight = screenY;
          bInfo.biPlanes = 1;
          bInfo.biBitCount = 24;
          bInfo.biCompression = BI_RGB;
          bInfo.biSizeImage = 0;
          bInfo.biClrUsed = 0;
          bInfo.biClrUsed = 0;

          Aztec::MImagePtr image = new Aztec::MImage();
          image->setSize(screenX, screenY);

          if (!GetDIBits(hdcCompatible, hbmScreen, 0, screenY, image->getPixelData(), (BITMAPINFO*)&bInfo, DIB_RGB_COLORS)) {
            errhandler("GetDIBits failed."); 
          } else {
            image->flipVertical();
            actualImage = new Aztec::MImage();
            actualImage->setSize(imageX, imageY);
            
            for (int y = 0; y < imageY; ++y) {
              for (int x = 0; x < imageX; ++x) {
                BYTE *from = image->getPixelData() + (y + viewRect.top) * (screenX * 3) + 3 * (x + viewRect.left);
                BYTE *to = actualImage->getPixelData() + y * imageX * 3 + x * 3;
                
                to[0] = from[2];
                to[1] = from[1];
                to[2] = from[0];
              }
            }
            
          }

        }
    
      }

      ::DeleteObject(hbmScreen);
    }


    ::DeleteDC(hdcCompatible);
    ::DeleteDC(hdcScreen);

    return actualImage;

#endif

  }

  static int viewCaptureToFile(const StringVector &args, std::string &result) {
    Aztec::MImagePtr image = captureCurrentView();

    return imageSaveAs(image) ? FunctionManager::SUCCEED : FunctionManager::FAIL;
  }

  static int viewCaptureToImageView(const StringVector &args, std::string &result) {
    Aztec::MImagePtr image = captureCurrentView();

    AztecViewPtr view = viewCreateNew("image");
    ImageViewPtr imageView = AZTEC_CAST(ImageView, view);

    if (imageView != NULL) {
      imageView->setImage(image);
    }


    return FunctionManager::FAIL;
  }

  static int viewToggleFullscreen(const StringVector &args, std::string &result) {
    AztecViewPtr view = AztecViewManager::getLastMouseOverView();

    if (view != NULL) {
      AztecViewManager::setCurrentView(view);
      // get the view's parent, and keep going upwards until we find a multi splitter. At that point, tell it to toggle the maximised state of the view.
      Aztec::MContainerPtr cont = view->getParent();;
      while (cont != NULL) {
        Aztec::MRefCountedPtr<AztecGUI::MultiSplitter> splitter = AZTEC_CAST(AztecGUI::MultiSplitter, cont);

        if (splitter != NULL) {
          splitter->toggleMaximisedComponent(view);
          return FunctionManager::SUCCEED;
        }
      }
    }

    return FunctionManager::FAIL;
  }


  static int viewCaptureRange(const StringVector &args, std::string &result) {
    // set up a save file dialog.
    Aztec::MSaveFileDialogPtr dlg = new Aztec::MSaveFileDialog();
    
    // add in all the filters from the scene translators
    Aztec::MPluginManagerPtr plugMan = Aztec::MSystemManager::getInstance()->getPluginManager();
    Aztec::MPluginManager::ImageTranslatorVector translators = plugMan->getImageTranslators(Aztec::MPluginManager::DONT_CARE, Aztec::MPluginManager::MUST);
    
    for (int i = 0; i < translators.size(); ++i) {
      dlg->addFilter(translators[i]->getFilterDescription(), translators[i]->getFilter());
    }

    dlg->setTitle("Save an image...");
    
    // Show the dialog, and if OK is pressed, so something useful.
    if (dlg->doModal()) {
      
      // The current filter for the dialog is >= 0 for manually picked filters, otherwise < 0 for automatically generated ones.
      if (dlg->getFilter() >= 0) {
        // If the user has picked a certain translator, use that translator, otherwise just use a general one.
        Aztec::MImageTranslatorPtr trans = translators[dlg->getFilter()];
        // if we have a translator, try to export the file.
        if (trans != NULL) {
          
          // now that we have a translator, extract the filename into parts, and then write out all the images.

          std::string fullname = dlg->getFilename();
          std::string name;
          std::string ext;
          int dot = fullname.rfind('.');
          if (dot != std::string::npos) {
            name = fullname.substr(0, dot);
            ext = fullname.substr(dot+1);
          } else {
            name = fullname;
          }

          Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
          double start = scene->tickToFrame(scene->getStartTime());
          double end = scene->tickToFrame(scene->getEndTime());
          int count = 0;
          for (double frame = start; frame <= end; frame += 0.5) {
            scene->setTime(scene->frameToTick(frame));
            AztecViewManager::redrawAllViews();

            Aztec::MImagePtr image = captureCurrentView();

            char thisFilename[1024];
            sprintf(thisFilename, "%s%03i.%s", name.c_str(), count, ext.c_str());
            ++count;
            trans->exportFile(thisFilename, image);
          }


          return FunctionManager::SUCCEED;
        }
      }
    }

    return FunctionManager::FAIL;
  }



  void registerViewFunctions(FunctionManager &man) {

    man.registerFunction("viewZoomIn", viewZoomIn);
    man.registerFunction("viewZoomOut", viewZoomOut);
    man.registerFunction("viewZoomToFitAll", viewZoomToFitAll);
    man.registerFunction("viewZoomToFitSelected", viewZoomToFitSelected);

    man.registerFunction("viewCreateCopy", viewCreateCopy);

    man.registerFunction("viewChangeCurrent", viewChangeCurrent);
    man.registerFunction("viewCreateNew", viewCreateNew);
    
    man.registerFunction("showWindow", showWindow);
    man.registerFunction("viewCaptureToFile", viewCaptureToFile);
    man.registerFunction("viewCaptureToImageView", viewCaptureToImageView);
    man.registerFunction("viewCaptureRange", viewCaptureRange);

    man.registerFunction("viewToggleFullscreen", viewToggleFullscreen);
  }


}

