#include <Aztec3DPCH.h>

// Aztec2 includes
#include <functions/FunctionManager.h>
#include <views/AztecViewManager.h>
#include <utils/SceneFunctions.h>

// AztecLib includes
#include <MSystemManager.h>
#include <MUIManager.h>
#include <MEditableMesh.h>
#include <misc/MSceneHelper.h>

// Standard includes
#include <algorithm>
#include <assert.h>

namespace AztecGUI {

  using namespace Aztec;

  template <typename A, typename B, typename C>
  static C findCommonValue(const A &a, const B &b, const C &defaultValue) {
    for (A::const_iterator i = a.begin(); i != a.end(); ++i) {
      for (B::const_iterator j = b.begin(); j != b.end(); ++j) {
        if (*i == *j) return *i;
      }
    }

    return defaultValue;
  }

  template <typename FUNC>
  static void pointConnectMesh(const Aztec::MEditableMeshPtr &mesh, FUNC condition) {

    // loop over the points to see if we can find a polygon with two selected points on it that doesn't have an edge between it.
    for (int v = 0; v < mesh->getNumVerts(); ++v) {
      if (!condition(mesh, v)) {
        continue;
      }

      std::vector<int> otherVerts;
      mesh->getVerticesConnecting(v, otherVerts);

      // loop over the connected vertices, get the polygon, and see if there 
      // is a selected point on it
      for (int connectedVert = 0; connectedVert < otherVerts.size(); ++connectedVert) {
        std::vector<int> polygon;
        mesh->getPolygonVerticesFromEdge(v, otherVerts[connectedVert], true, polygon);

        int otherPoint = -1;
        for (int p = 0; p < polygon.size(); ++p) {
          if (polygon[p] != v && condition(mesh, polygon[p])) {
            otherPoint = polygon[p];
            break;
          }
        }

        // if we have a point to connect to, see if there is an edge connecting them.
        if (otherPoint != -1) {
          std::vector<int> connectedTris;

          mesh->getTrianglesWithEdge(v, otherPoint, connectedTris);
          // if we have a triangle already, all we have to do is make the edge visible.
          
          if (connectedTris.size() > 0) {
            mesh->setEdgeFlag(v, otherPoint, EDGE_VISIBLE);
          } else {
            mesh->unsetTriangleFlags(TRIANGLE_FLAGFORCHANGE);

            // otherwise we have to retriangulate the polygon.
            mesh->getTrianglesWithEdge(v, otherVerts[connectedVert], connectedTris);
            
            // now create our two new polygons.
            std::vector<int> newPoly1, newPoly2;
            std::vector<int> *out1 = &newPoly1;
            std::vector<int> *out2 = &newPoly2;

            for (int i = 0; i < polygon.size(); ++i) {
              out1->push_back(polygon[i]);

              // if we hit our other point switch over to add points to the new polygon.
              if (polygon[i] == otherPoint) {
                std::swap(out1, out2);
                out1->push_back(otherPoint);
              }
            }

            // add on our last vertex.
            out1->push_back(v);

            // if we have valid new polygons, delete the old ones, and create the new.
            if (newPoly1.size() >= 3 && newPoly2.size() >= 3) {
              std::vector<int> allVerts;
              mesh->getVerticesConnecting(polygon[1], allVerts, true);

              // find our start 
              int start = -1;
              for (int i = 0; i < allVerts.size(); ++i) {
                if (allVerts[i] == v) {
                  start = i;
                }
              }

              if (start != -1) {
                std::vector<int> tris1, tris2;
                mesh->getTrianglesWithEdge(polygon[1], allVerts[start], tris1);
                mesh->getTrianglesWithEdge(polygon[1], allVerts[(start + 1) % allVerts.size()], tris2);

                int commonTri = findCommonValue(tris1, tris2, -1);

                assert(commonTri != -1);

                mesh->flagPolygon(commonTri, TRIANGLE_FLAGFORCHANGE);
//                mesh->flagPolygon(commonTri, TRIANGLE_SELECTED);

                // delete the polygon
                mesh->deleteTriangleFlag(TRIANGLE_FLAGFORCHANGE);

                std::reverse(newPoly1.begin() + 1, newPoly1.end());
                std::reverse(newPoly2.begin() + 1, newPoly2.end());

                mesh->triangulatePolygon(newPoly1);
                mesh->triangulatePolygon(newPoly2);
              }
            }

          }


        }
      }


    }
    
    
  }

  static bool selectedCondition(const Aztec::MEditableMeshPtr &mesh, int vertex) {
    return mesh->isVertexFlagged(vertex, VERTEX_SELECTED);
  }

  class twoPointsCondition {
  public:
    twoPointsCondition(int p1, int p2) {
      a = p1;
      b = p2;
    }

    bool operator()(const Aztec::MEditableMeshPtr &mesh, int vertex) { 
      return (vertex == a) || (vertex == b);
    }

    int a;
    int b;
  };

  void pointConnectMesh(const Aztec::MEditableMeshPtr &mesh) {
    pointConnectMesh(mesh, selectedCondition);
  }

  void pointConnectMesh(const Aztec::MEditableMeshPtr &mesh, int a, int b) {
    pointConnectMesh(mesh, twoPointsCondition(a,b));
  }

  static void doPointConnect(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    MNamedObjectPtr namedObj = AZTEC_CAST(MNamedObject, node->getObject());
    MEditableMeshPtr mesh = getEditableMeshObject(namedObj);
    if (mesh == NULL) {
      MSystemManager::getInstance()->logOutput("Warning: pointConnect() failed on object '%s', only operates on editable meshes", namedObj != NULL ? namedObj->getName().c_str() : "");
      return;
    }

    if (mesh->getComponentMode() == MComponentisedObject::POINT_TYPE) {
      pointConnectMesh(mesh);
    }
  }

  int pointConnect(const StringVector &args, std::string &result) {
    applyToObjects(MScene::getGlobalScene(), 
                   MScene::getGlobalScene()->getObjectList(), 
                   ifSelectedCriteria, 
                   doPointConnect);

    AztecViewManager::redrawAllViews();
	  
    return FunctionManager::SUCCEED;
  }


}

