#include <Aztec3DPCH.h>

#include <gui/MSaveFileDialog.h>
#include <gui/MOpenFileDialog.h>
#include <gui/MMessageDialog.h>

#include <config/UIConfig.h>
#include <controls/MainWindow.h>
#include <functions/file/FileFunctions.h>
#include <views/AztecViewManager.h>
#include <views/ImageView.h>

#include <MScene.h>
#include <MSystemManager.h>
#include <MCompleteTranslator.h>

namespace AztecGUI {

  static int sceneSave(const StringVector &args, std::string &result);
  static int sceneSaveAs(const StringVector &args, std::string &result);
  static int sceneQuit(const StringVector &args, std::string &result);
  static int sceneNew(const StringVector &args, std::string &result);

  /**
   * This checks to see if we have to save any changes. If changes have been 
   * made, then a dialog is presenting, asking if we wish to save the file. 
   * If the user picks yes, then the file is saved, and the function returns 
   * true. If the user picks no, the file isn't and the function returns 
   * true. If the user cancels, the function does nothing and returns false.
   *
   * @return true if the calling function should continue processing, false if 
   *         the function should cancel.
   */
  static bool checkForSaveChanges() {
    if (UIConfig::changesMade()) {
      Aztec::MMessageDialogPtr dlg = new Aztec::MMessageDialog();

      Aztec::MMessageDialog::Buttons result = dlg->doModal(NULL, "Save Changes", "Do you wish to save your changes?", Aztec::MMessageDialog::YES | Aztec::MMessageDialog::NO | Aztec::MMessageDialog::CANCEL);
      
      if (result == Aztec::MMessageDialog::CANCEL) {
        // we only want to return false if we cancelled. All other situations return true.
        return false;
      }

      if (result == Aztec::MMessageDialog::YES) {
        std::string result;
        // if the save file function had an error, make sure we cancel the 
        // operation, as the user may want to fix things.
        if (sceneSave(StringVector(), result) != FunctionManager::SUCCEED) {
          return false;
        }
      }
    }

    return true;
  }

  static int sceneQuit(const StringVector &args, std::string &result) {
    // try to quit
    if (MainWindow::getInstance() != NULL) {
      MainWindow::getInstance()->close();
    }
    return FunctionManager::SUCCEED;
  }

  static int sceneNew(const StringVector &args, std::string &result) {
    if (!checkForSaveChanges()) {
      return FunctionManager::FAIL;
    }

    // now clear the scene out.
    Aztec::MScene::getGlobalScene()->clearScene();
    Aztec::MScene::getGlobalScene()->setTime(0);
    UIConfig::setCurrentFilename("");

    // update the displays.
    AztecViewManager::redrawAllViews();

    return FunctionManager::SUCCEED; 
  }

  static int sceneSave(const StringVector &args, std::string &result) {
    return sceneSaveAs(args, result);

    // if we have no current filename, then do a save as instead
    if (UIConfig::getCurrentFilename().length() == 0) {
      return sceneSaveAs(args, result);
    } else {

      // TODO: Save some actual scene data.

      return FunctionManager::FAIL;
    }
  }

  static int sceneSaveAs(const StringVector &args, std::string &result) {
    Aztec::MSaveFileDialogPtr dlg = new Aztec::MSaveFileDialog;

    dlg->addFilter("Aztec Scene Files", "*.AztecAscii");
    dlg->setTitle("Save a scene...");

    if (dlg->doModal()) {
      Aztec::MCompleteTranslator trans;
      trans.exportFile(dlg->getFilename().c_str(), Aztec::MScene::getGlobalScene());
    }

    return FunctionManager::FAIL;
  }

  static int sceneOpen(const StringVector &args, std::string &result) {
    if (!checkForSaveChanges()) {
      return FunctionManager::FAIL;
    }

    // try to open the file.
    Aztec::MOpenFileDialogPtr dlg = new Aztec::MOpenFileDialog;

    dlg->addFilter("Aztec Scene Files", "*.AztecAscii");
    dlg->setTitle("Open a scene...");

    // Show the dialog, and if OK is pressed, so something useful.
    if (dlg->doModal()) {
      // Get pur global scene
      Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();

      // clear out the scene first.
      scene->clearScene();

      // Now import our file.
      Aztec::MCompleteTranslator trans;
      bool result = trans.importFile(dlg->getFilename().c_str(), scene);

      // update the displays.
      AztecViewManager::redrawAllViews();

      return result ? FunctionManager::SUCCEED : FunctionManager::FAIL;
    } else {
      return FunctionManager::FAIL;
    }
  }

  static int sceneImport(const StringVector &args, std::string &result) {
    // try to open the file.
    Aztec::MOpenFileDialogPtr dlg = new Aztec::MOpenFileDialog;

    // add in all the filters from the scene translators
    Aztec::MPluginManagerPtr plugMan = Aztec::MSystemManager::getInstance()->getPluginManager();
    Aztec::MPluginManager::SceneTranslatorVector translators = plugMan->getSceneTranslators(Aztec::MPluginManager::MUST);

    for (int i = 0; i < translators.size(); ++i) {
      dlg->addFilter(translators[i]->getFilterDescription(), translators[i]->getFilter());
    }

    dlg->setTitle("Import a scene...");

    // Show the dialog, and if OK is pressed, so something useful.
    if (dlg->doModal()) {
      // If the user has picked a certain translator, use that translator, otherwise just use a general one.
      Aztec::MSceneTranslatorPtr trans;
      // The current filter for the dialog is >= 0 for manually picked filters, otherwise < 0 for automatically generated ones.
      if (dlg->getFilter() >= 0) {
        trans = translators[dlg->getFilter()];
      } else {
        trans = plugMan->getSceneTranslatorThatImports(dlg->getFilename().c_str());
      }

      // if we have a translator, try to import the file.
      if (trans != NULL) {
        // Get pur global scene
        Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();

        // now try to import it.
        bool result = trans->importFile(dlg->getFilename().c_str(), scene);

        // update the displays.
        AztecViewManager::redrawAllViews();
        
        return result ? FunctionManager::SUCCEED : FunctionManager::FAIL;
      } else {
        // if we have a null translator, we have failed.
        return FunctionManager::FAIL;
      }

    } else {

      // otherwise we have failed
      return FunctionManager::FAIL;
    }
  }

  static int sceneExport(const StringVector &args, std::string &result) {
    // try to open the file.
    Aztec::MSaveFileDialogPtr dlg = new Aztec::MSaveFileDialog;

    // add in all the filters from the scene translators
    Aztec::MPluginManagerPtr plugMan = Aztec::MSystemManager::getInstance()->getPluginManager();
    Aztec::MPluginManager::SceneTranslatorVector translators = plugMan->getSceneTranslators(Aztec::MPluginManager::DONT_CARE, Aztec::MPluginManager::MUST);

    for (int i = 0; i < translators.size(); ++i) {
      dlg->addFilter(translators[i]->getFilterDescription(), translators[i]->getFilter());
    }

    dlg->setTitle("Export a scene...");

    if (dlg->doModal()) {
      int filter = dlg->getFilter();
      Aztec::MSceneTranslatorPtr trans = translators[filter];
      trans->exportFile(dlg->getFilename().c_str(), Aztec::MSystemManager::getInstance()->getScene());
    }

    return FunctionManager::FAIL;
  }

  static int imageViewOpenImage(const StringVector &args, std::string &result) {
    ImageViewPtr imageView = AZTEC_CAST(ImageView, AztecViewManager::getCurrentView());

    if (imageView == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: imageViewOpenImage() - current view needs to be an image view!");
      return FunctionManager::FAIL;

    } else {

      if (imageView->openImage()) {
        return FunctionManager::SUCCEED;
      } else {
        return FunctionManager::FAIL;
      }

    }
    
  }

  static int imageViewSaveImageAs(const StringVector &args, std::string &result) {
    ImageViewPtr imageView = AZTEC_CAST(ImageView, AztecViewManager::getCurrentView());

    if (imageView == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: imageViewOpenImage() - current view needs to be an image view!");
      return FunctionManager::FAIL;

    } else {

      if (imageView->saveImageAs()) {
        return FunctionManager::SUCCEED;
      } else {
        return FunctionManager::FAIL;
      }

    }
  }

  bool imageSaveAs(const Aztec::MImagePtr image) {
    // try and save an image.

    // if we have no image to save, we can do nothing.
    if (image == NULL) {
      return false;
    }
    
    // set up a save file dialog.
    Aztec::MSaveFileDialogPtr dlg = new Aztec::MSaveFileDialog();
    
    // add in all the filters from the scene translators
    Aztec::MPluginManagerPtr plugMan = Aztec::MSystemManager::getInstance()->getPluginManager();
    Aztec::MPluginManager::ImageTranslatorVector translators = plugMan->getImageTranslators(Aztec::MPluginManager::DONT_CARE, Aztec::MPluginManager::MUST);
    
    for (int i = 0; i < translators.size(); ++i) {
      dlg->addFilter(translators[i]->getFilterDescription(), translators[i]->getFilter());
    }

    dlg->setTitle("Save an image...");
    
    // Show the dialog, and if OK is pressed, so something useful.
    if (dlg->doModal()) {
      
      // The current filter for the dialog is >= 0 for manually picked filters, otherwise < 0 for automatically generated ones.
      if (dlg->getFilter() >= 0) {
        // If the user has picked a certain translator, use that translator, otherwise just use a general one.
        Aztec::MImageTranslatorPtr trans = translators[dlg->getFilter()];
        // if we have a translator, try to export the file.
        if (trans != NULL) {
         
          // now try to export it.
          bool result = trans->exportFile(dlg->getFilename().c_str(), image);
          
          if (result) {
            Aztec::MSystemManager::getInstance()->logOutput("Saved image '%s'", dlg->getFilename().c_str());
          } else {
            Aztec::MSystemManager::getInstance()->logOutput("Error: failed to save image '%s'", dlg->getFilename().c_str());
          }
          
          return result;
        } else {
          return false;
        }
      }
      
    }

    // if we have gotten here, everything has failed, so return out.
    return false;
  }


  void registerFileFunctions(FunctionManager &man) {
    man.registerFunction("quit", sceneQuit);
    man.registerFunction("sceneNew", sceneNew);
    man.registerFunction("sceneSave", sceneSave);
    man.registerFunction("sceneSaveAs", sceneSaveAs);
    man.registerFunction("sceneOpen", sceneOpen);
    man.registerFunction("sceneImport", sceneImport);
    man.registerFunction("sceneExport", sceneExport);
    man.registerFunction("imageViewOpenImage", imageViewOpenImage);
    man.registerFunction("imageViewSaveImageAs", imageViewSaveImageAs);
  }

}

