#include <Aztec3DPCH.h>

// Aztec2 includes
#include <controls/GraphComponent.h>

#include <views/AztecGraphView.h>
#include <views/AztecViewManager.h>
#include <tools/MGraphSelectTool.h>
#include <utils/AztecGLUtils.h>
#include <tools/MToolManager.h>

// AztecLib includes
#include <MScene.h>
#include <MUIManager.h>
#include <MSystemManager.h>
#include <scripting/MScriptInterp.h>

// standard includes
#include <math.h>


namespace AztecGUI {

  GraphComponent::GraphComponent(AztecView *parentView) 
    : AztecGLFontCanvas()
  {
    m_Dragging = false;
    minTick = MScene::getGlobalScene()->getStartTime();
    maxTick = MScene::getGlobalScene()->getEndTime();
    minValue = -100;
    maxValue = 100;
    this->parentView = parentView;
  }

  GraphComponent::~GraphComponent()
  {

  }

  void GraphComponent::renderXAxis() const {
    // TODO: thid should be changed so it can handle things other than
    // time as an input. There is no rush for this, as this will only be necessary
    // when KeyList's can have things other than time as an input.

    // draw the current time.
    glColor4f(0.1f, 0.4f, 0.9f,0.5f);
    glBegin(GL_QUADS);
    glVertex2f(MScene::getGlobalScene()->getTime(), maxValue);
    glVertex2f(MScene::getGlobalScene()->getTime() + MScene::getGlobalScene()->getTicksPerFrame(), maxValue);
    glVertex2f(MScene::getGlobalScene()->getTime() + MScene::getGlobalScene()->getTicksPerFrame(), minValue);
    glVertex2f(MScene::getGlobalScene()->getTime(), minValue);
    glEnd();

    // Draw the guides
    glBegin(GL_LINES);

    glColor3f(0,0,0);
    glVertex2f(minTick,maxValue);
    glVertex2f(minTick,minValue);

    glVertex2f(minTick,0);
    glVertex2f(maxTick,0);

    glEnd();


    // Draw the Timelines Text
    long frameMin, frameMax;

    frameMin = MScene::getGlobalScene()->tickToFrame(minTick);
    frameMax = MScene::getGlobalScene()->tickToFrame(maxTick);

    int frameStep = 1;
    {
      bool collision = true;
#ifdef WIN32
      HDC dc = ::GetDC(m_hWnd);

      while (collision) {
        RECT lastTextRect, textRect;
        SIZE size;

        lastTextRect.left = -5000;
        lastTextRect.right = -5000;
        lastTextRect.top = -10;
        lastTextRect.bottom = -10;

        collision = false;
        if (frameMin + frameStep <= frameMax) {
          for (int frame = frameMin; frame <= frameMax; frame += frameStep) {
            int tick = MScene::getGlobalScene()->frameToTick(frame);
            float xPos, yPos;
            clientToGraph(0,15, xPos, yPos);

            MStr str;
            str.Format("%i", frame);
            ::GetTextExtentPoint32(dc, (LPCTSTR)str, str.GetLength(), &size);

            graphToClient(tick, yPos, textRect.left, textRect.bottom);

            textRect.right = textRect.left + size.cx + 3;
            textRect.top = textRect.bottom - size.cy;

            if (textRect.left < lastTextRect.right) {
              collision = true;
              break;
            }

            lastTextRect = textRect;
          }
        }

        if (collision) {
          if (frameStep == 1) {
            frameStep = 2;
          } else if (frameStep == 2) {
            frameStep = 5;
          } else if (frameStep < 30) {
            frameStep += 5;
          } else if (frameStep < 50) {
            frameStep += 20;
          } else {
            frameStep += 50;
          }
        }
      }

      ::ReleaseDC(m_hWnd, dc);
#else
#warning TODO: Linux code
#endif
    }

    glLineWidth(1);

    // now we adjust our frame min to ensure that the frameStep will
    // fall onto the 0 mark.
    if (frameMin < 0) {
      int diff = (frameStep-frameMin) % frameStep;
      frameMin -= (frameStep - diff);
    }


    // now we render our actual time line.
    for (int frame = frameMin; frame <= frameMax; frame += frameStep) {
      int tick = MScene::getGlobalScene()->frameToTick(frame);

      glColor3f(0.3f,0.3f,0.3f);
      glBegin(GL_LINES);
      glVertex2f(tick, maxValue);
      glVertex2f(tick, minValue);
      glEnd();

      float fx, fy;
      int ix, iy;
      clientToGraph(0,15, fx, fy);
      graphToClient(tick, fy, ix, iy);
      clientToGraph(ix + 2, iy , fx, fy);

      glColor3f(1,1,1);
      glDrawText(fx, fy, Aztec::MUIManager::tickCountToTimeStamp(MScene::getGlobalScene()->frameToTick(frame)), getFont());
    
    }
  }

#ifdef WIN32
  void GraphComponent::getKeysInRect(const RECT &rect, std::vector<ListKey> &keys) {
    keys.clear();
  
    float minValue, maxValue;
    float minTime, maxTime;

    clientToGraph(rect.left, rect.top, minTime, minValue);
    clientToGraph(rect.right, rect.bottom, maxTime, maxValue);

    if (minTime > maxTime) {
      std::swap(minTime, maxTime);
    }
    if (minValue > maxValue) {
      std::swap(minValue, maxValue);
    }

    // go through out current value lits, and try to see if any keys are underneath it.
    for (int i = 0; i < graphValues.size(); ++i) {
      MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);
    
      if (keyList != NULL) {
        // loop over the keys
        for (int keyIndex = 0; keyIndex < keyList->getKeyCount(); ++keyIndex) {
          float value;
          long time;

          time = keyList->getTimeAtIndex(keyIndex);
          value = keyList->getValueAtIndex(keyIndex);

          if (time >= minTime && time <= maxTime && 
              value >= minValue && value <= maxValue) {

            keys.push_back(std::make_pair(keyList, keyIndex));
          }
        }
      }
    }
  }
#else
#warning TODO: getKeysInRect method for Linux version
#endif

  void GraphComponent::getKeysAtPoint(int x, int y, std::vector<ListKey> &keys, int radius) {
#ifdef WIN32
    RECT rect;

    rect.left = x - radius;
    rect.right = x + radius;
    rect.top = y - radius;
    rect.bottom = y + radius;

    getKeysInRect(rect, keys);
#else
#warning TODO: getKeysAtPoint method for Linux version
#endif
  }

  MFloatKeyListPtr GraphComponent::getKeyListAtMouse(int mouseX, int mouseY) {
    for (int i = 0; i < graphValues.size(); ++i) {
      MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);

      if (keyList == NULL) {
        continue;
      }

      float x1,y1;
      float minY,maxY;

      clientToGraph(mouseX, mouseY+3, x1, minY);
      clientToGraph(mouseX, mouseY-3, x1, maxY);

      y1 = keyList->getValueAtTime(x1);

      // if we have a value that falls within our range, this is our baby.
      if (y1 >= minY && y1 <= maxY) {
        return keyList;
      }
    }

    return NULL;
  }

  void GraphComponent::getGraphRange(float &min, float &max) {
    min = minTick;
    max = maxTick;
  }

  void GraphComponent::setGraphRange(float min, float max) {

    // only bother zooming in if it will make a difference
    if (max - min > MScene::getGlobalScene()->getTicksPerFrame()) {
      minTick = min;
      maxTick = max;
    }
  }

  void GraphComponent::getValueRange(float &min, float &max) {
    min = minValue;
    max = maxValue;
  }

  void GraphComponent::setValueRange(float min, float max) {
    minValue = min;
    maxValue = max;
  }

  void GraphComponent::initialiseRendering() {
    makeCurrent();
  
    glEnable(GL_POINT_SMOOTH);
    glDisable(GL_LINE_SMOOTH);

    glDisable(GL_DEPTH_TEST);
    glDisable(GL_CULL_FACE);
  
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glPointSize(4);
  
    // set up the drawing modes
  
    glLineWidth(2);

#ifdef WIN32
    RECT clientRect;
    ::GetClientRect(m_hWnd, &clientRect);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glViewport(0,0,clientRect.right, clientRect.bottom);
    glTranslatef(-1.0f,0.0f,0.0f);
    glScalef(2.0/(maxTick-minTick), 2.0/(maxValue - minValue), 1);
    glTranslatef(-minTick,-(minValue + maxValue)/2.0,0);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
#else
#warning TODO: Linux code
#endif
    glClearColor(0.4f, 0.4f, 0.4f, 0.0f);
    glClear(GL_COLOR_BUFFER_BIT);
  }
  //  minTick / maxTick - minTick
  void GraphComponent::draw3D() {
    MSize2D clientRect = getClientSize();
    if (clientRect.getWidth() <= 1 || clientRect.getHeight() <= 1) {
      return;
    }

    initialiseRendering();

    renderXAxis();
    renderYAxis();

    {
#ifdef WIN32
      MBaseObjectTreePtr Tree = MScene::getGlobalScene()->getObjectList();
      MBaseObjectPtr BaseObj;
      MSceneObjectPtr Obj;
      RECT clientRect;
    
      ::GetClientRect(m_hWnd, &clientRect);

      for (int i = 0; i < graphValues.size(); ++i) {

        MFloatValuePtr value = graphValues[i].value;
        MVector3 colour = graphValues[i].colour;

        float timeStep;
        timeStep = (float)(maxTick - minTick) / (clientRect.right/2);

        glColor3f(colour.x, colour.y, colour.z);

        glBegin(GL_LINE_STRIP);
        float pos;
        for (float time = minTick; time < maxTick; time += timeStep) {
          pos = value->getValueAtTime(time);
          glVertex2f(time, pos);
        }
        glEnd();

        // Draw the control points on each spline.
        glPointSize(8);

        glBegin(GL_POINTS);
        MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, value);

        if (keyList != NULL) {
          for (int listIndex = 0; listIndex < 3; ++listIndex) {
            for (int index = 0; index < keyList->getKeyCount(); ++index) {
              float pos;
              pos = keyList->getValueAtIndex(index);
              MKeyPtr key = keyList->getKeyAtIndex(index);

              if (key->isSelected()) {
                glColor3f(0.9f,0.9f,0.9f);
              } else {
                glColor3f(0.7f,0.7f,0.7f);
              }

              glVertex2f(key->getKeyTime(), pos);
            }
          }
          glEnd();
      
          glPointSize(4);
          glBegin(GL_POINTS);
          for (int listIndex = 0; listIndex < 3; ++listIndex) {
            for (int index = 0; index < keyList->getKeyCount(); ++index) {
              float pos;
              pos = keyList->getValueAtIndex(index);
              MKeyPtr key = keyList->getKeyAtIndex(index);

              if (key->isSelected()) {
                glColor3f(1,1,0);
              } else {
                glColor3f(0.2f,0.2f,0.2f);
              }

              glVertex2f(key->getKeyTime(), pos);
            }
          }
          glEnd();
        }

      }
#else
#warning TODO: Linux code
#endif
    }

    MToolTypePtr tool = MToolManager::getInstance()->GetTool("graphView");

    if (tool != NULL) {
      tool->drawTool(false, parentView);
    }

    glFlush();
    glFinish();
  
    swapBuffers();

  }

  // MComponent methods
  bool GraphComponent::onPaint() {
    AztecGLFontCanvas::onPaint();
    draw3D();

    return true;
  }

  bool GraphComponent::onMousePressed(const Aztec::MMouseEvent &event) {
    Aztec::MSystemManager::getInstance()->getUndoManager()->beginUndo("Graph Edit");
    setMouseCapture(true);
    setAsCurrentView();

    MToolTypePtr tool = MToolManager::getInstance()->GetTool("graphView");

    if (tool != NULL) {
      if (tool->RequiresSelection()) {
        bool doSelection = false;

        if (isAnythingSelected()) {
          // make sure that we have a key underneath us
          std::vector<ListKey> keys;
          getKeysAtPoint(event.getX(), event.getY(), keys);

          // set it up so we assume there are no selected keys underneath our cursor
          doSelection = true;

          // now loop over the keys under our mouse, and see if any are selected.
          // If one or more are selected, then we don't need to switch to the
          // selection tool, so we break out of the loop.
          for (int i = 0; i < keys.size(); ++i) {
            if (keys[i].first->getKeyAtIndex(keys[i].second)->isSelected()) {
              doSelection = false;
              break;
            }
          }
        } else {
          doSelection = true;
        }

        if (doSelection) {
          MToolManager::getInstance()->setTemporary(getSelectTool(), "graphView");
          tool = MToolManager::getInstance()->GetTool("graphView");
        }
      }

      parentView->handleToolResult(tool->onMouseDown(event));
    }

    return true;
  }

  bool GraphComponent::onMouseReleased(const Aztec::MMouseEvent &event) {
    if (event.getShiftState().isMouseUp()) {
      setMouseCapture(false);
    }
    MToolTypePtr tool = MToolManager::getInstance()->GetTool("graphView");

    if (tool != NULL) {
      parentView->handleToolResult(tool->onMouseUp(event));

      MToolManager::getInstance()->unsetTemporary("graphView");
    }

    Aztec::MSystemManager::getInstance()->getUndoManager()->endUndo();
    return true;
  }

  bool GraphComponent::onMouseMove(const Aztec::MMouseEvent &event) {
    setMouseCursor(Aztec::Mouse_Normal);
    MToolTypePtr tool = MToolManager::getInstance()->GetTool("graphView");

    if (tool != NULL) {
      parentView->handleToolResult(tool->onMouseMove(event));
    }
    return true;
  }

  bool GraphComponent::onMouseWheel(const Aztec::MMouseEvent &event) {
    if (event.getType() == Aztec::MMouseEvent::WHEEL_UP) {
      Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.viewZoomIn();");
      return true;
    } else if (event.getType() == Aztec::MMouseEvent::WHEEL_DOWN) {
      Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.viewZoomOut();");
      return true;
    } else {
      return false;
    }

  }

  MToolTypePtr GraphComponent::getSelectTool() {
    return new MGraphSelectTool();
  }

  bool GraphComponent::isAnythingSelected() {
    // go through out current value lits, and try to see if any keys are underneath it.
    for (int i = 0; i < graphValues.size(); ++i) {
      MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);
    
      if (keyList != NULL) {
        // loop over the keys
        for (int keyIndex = 0; keyIndex < keyList->getKeyCount(); ++keyIndex) {
          if (keyList->getKeyAtIndex(keyIndex)->isSelected()) {
            return true;
          }
        }
      }
    }
    return false;
  }

  void GraphComponent::setGraphValues(const std::vector<GraphPair> &values) {
    graphValues = values;
  }

  void GraphComponent::getGraphValues(std::vector<GraphPair> &values) {
    values = graphValues;
  }

  bool GraphComponent::isCurrentView() {

    return AztecViewManager::getCurrentView() == getParent();
  }

  void GraphComponent::setAsCurrentView() {

    AztecViewPtr parent = AztecView::getViewForComponent(this);

    if (parent != NULL) {
      AztecViewManager::setCurrentView(parent);
    }
  }

/*  DWORD GraphComponent::ViewPopupMenu(int x, int y) {
    CGraphViewWnd *parent = dynamic_cast<CGraphViewWnd*>(GetParent());

    if (parent != NULL) {
      parent->ViewPopupMenu(x,y);
    }

    return 0;
  }

  int GraphComponent::HandlePopupCommand(DWORD Cmd) {
    CGraphViewWnd *parent = dynamic_cast<CGraphViewWnd*>(GetParent());

    if (parent != NULL) {
      parent->HandlePopupCommand(Cmd);
    }

    return 0;
  }
*/
  void GraphComponent::renderYAxis() const {
    // Draw the guides
    glBegin(GL_LINES);

    glColor3f(0,0,0);
  //  glVertex2f(minTick,maxValue);
  //  glVertex2f(minTick,minValue);

    glVertex2f(0, minValue);
    glVertex2f(0, maxValue);

    glEnd();

    const float stepsToUse[]  = { 0.01f , 0.05f , 0.1f  , 0.2f  , 0.5f  , 1.0f  , 2.0f  , 5.0f  , 10.0f , 20.0f , 25.0f , 50.0f };
    const char *formatToUse[] = { "%.2f", "%.2f", "%.1f", "%.1f", "%.1f", "%.0f", "%.0f", "%.0f", "%.0f", "%.0f", "%.0f", "%.0f" };
    const int MAX_NUM_STEPS = 12;
    const char *valueFormat = formatToUse[0];
    bool cantDraw = false;

    // Draw the Timelines Text
    float valueStep = stepsToUse[0];
    {
#ifdef WIN32
      bool collision = true;
      int collisionCount = 0;
      HDC dc = ::GetDC(m_hWnd);

      while (collision) {
        RECT lastTextRect, textRect;
        SIZE size;

        lastTextRect.left = 0;
        lastTextRect.right = 10;
        lastTextRect.top = 5000;
        lastTextRect.bottom = 5000;

        collision = false;
        if (minValue + valueStep <= maxValue) {
          for (float value = minValue; value <= maxValue; value += valueStep) {
            float xPos, yPos;
            clientToGraph(0,15, xPos, yPos);

            MStr str;
            str.Format(valueFormat, value);
            ::GetTextExtentPoint32(dc, (LPCTSTR)str, str.GetLength(), &size);

            graphToClient(xPos, value, textRect.left, textRect.bottom);

            textRect.right = textRect.left + size.cx + 3;
            textRect.top = textRect.bottom - size.cy;

            // since the Y Axis will be drawn going up the screen we have to 
            // check to see if the bottom of the new rect is greater that the
            // top of the last rect.
            if (textRect.bottom >= lastTextRect.top) {
              collision = true;
              break;
            }

            lastTextRect = textRect;
          }
        }

        if (collision) {
          ++collisionCount;
          if (collisionCount >= MAX_NUM_STEPS) {
            collisionCount = MAX_NUM_STEPS - 1;
            valueStep += stepsToUse[collisionCount];
          } else {
            valueStep = stepsToUse[collisionCount];
          }

          valueFormat = formatToUse[collisionCount];

          // if we have a collision, but our step is great than our window 
          // height, we just return out o this function, because we can't 
          // possibly draw anything when things are like that.
          if (valueStep > maxValue - minValue) {
            cantDraw = true;
            break;
          }

        }
      }

      ::ReleaseDC(m_hWnd, dc);
#else
#warning TODO: Linux code
#endif
    }

    if (cantDraw) {
      return;
    }

    glLineWidth(1);

    float valueMinToUse = minValue;
    // now we adjust our frame min to ensure that the frameStep will
    // fall onto the 0 mark.
    if (valueMinToUse < 0) {
      float diff = fmod(valueStep - minValue, valueStep);
      valueMinToUse -= (valueStep - diff);
    }


    // now we render our actual time line.
    for (float value = valueMinToUse; value <= maxValue; value += valueStep) {
      glColor3f(0.3f,0.3f,0.3f);
      glBegin(GL_LINES);
      glVertex2f(minTick, value);
      glVertex2f(maxTick, value);
      glEnd();

      float fx, fy;
      int ix, iy;
      clientToGraph(0,15, fx, fy);
      graphToClient(fx, value, ix, iy);
      clientToGraph(ix + 2, iy , fx, fy);

      // don't draw graph values if they are near the top.
      if (iy < 35) {
        continue;
      }

      MStr str;
      str.Format(valueFormat, value);

      glColor3f(1,1,1);
      glDrawText(fx, fy, str.c_str());
    
    }
  }

/*  void GraphComponent::OnHScroll( UINT nSBCode, UINT nPos, CScrollBar* pScrollBar ) {

    float diff = 0;

    switch (nSBCode) {
    case SB_LINELEFT: 
      diff = -0.1*(maxTick - minTick);
      if (diff == 0) diff = -MScene::getGlobalScene()->getTicksPerFrame();
      break;
    case SB_LINERIGHT: 
      diff = 0.1*(maxTick - minTick);
      if (diff == 0) diff = MScene::getGlobalScene()->getTicksPerFrame();
      break;
    case SB_PAGELEFT: 
      diff = -0.9*(maxTick - minTick);
      if (diff == 0) diff = -MScene::getGlobalScene()->getTicksPerFrame();
      break;
    case SB_PAGERIGHT: 
      diff = 0.9*(maxTick - minTick);
      if (diff == 0) diff = MScene::getGlobalScene()->getTicksPerFrame();
      break;
    }

    if (diff != 0) {
      minTick += diff;
      maxTick += diff;
      DrawView();
    }
  }

  void GraphComponent::OnVScroll( UINT nSBCode, UINT nPos, CScrollBar* pScrollBar ) {

    float diff = 0;

    switch (nSBCode) {
    case SB_LINELEFT: 
      diff = 0.1*(maxValue - minValue);
      if (diff == 0) diff = 0.1f;
      break;
    case SB_LINERIGHT: 
      diff = -0.1*(maxValue - minValue);
      if (diff == 0) diff = -0.1f;
      break;
    case SB_PAGELEFT: 
      diff = 0.9*(maxValue - minValue);
      if (diff == 0) diff = 1.0f;
      break;
    case SB_PAGERIGHT: 
      diff = -0.9*(maxValue - minValue);
      if (diff == 0) diff = -1.0f;
      break;
    }

    if (diff != 0) {
      minValue += diff;
      maxValue += diff;
      DrawView();
    }
  }

  BOOL GraphComponent::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt) {
    float middle = 0.5*(maxTick + minTick);
    float diff = maxTick - minTick;
    if (zDelta < 0) {
      diff *= 0.5;

      if (diff < 2) {
        diff = 2;
      }

    } else if (zDelta > 0) {
      diff *= 2;
    }
  
    minTick = middle - diff/2;
    maxTick = middle + diff/2;
    DrawView();
  
    return CWnd::OnMouseWheel(nFlags, zDelta, pt);
  }
*/
  void GraphComponent::clientToGraph(int ix, int iy, float &fx, float &fy) const {
#ifdef WIN32
    float glX, glY;

    RECT clientRect;

    ::GetClientRect(m_hWnd, &clientRect);

    glX = 2.0 * (float)ix / (float)clientRect.right - 1.0;
    glY = -2.0 * (float)iy / (float)clientRect.bottom + 1.0;

    fx = (glX + 1.0) / (2.0 / (maxTick - minTick)) + minTick;
    fy = glY / (2.0 / (maxValue - minValue)) + (minValue + maxValue)/2.0;
#else
#warning TODO: Linux code
#endif
  }

  void GraphComponent::graphToClient(float fx, float fy, int &ix, int &iy) const {
    float glX, glY;

    MSize2D clientRect = const_cast<GraphComponent*>(this)->getClientSize();

    glX = (fx - minTick) * (2.0 / (maxTick - minTick)) - 1.0;
    glY = (fy - (minValue + maxValue)/2.0) * (2.0 / (maxValue - minValue));

    ix = (glX + 1.0) / (2.0 / (float)clientRect.getRight());
    iy = (glY - 1.0) / (-2.0 / (float)clientRect.getBottom());
  }

  void GraphComponent::graphToClient(float fx, float fy, long &ix, long &iy) const {
    float glX, glY;

    MSize2D clientRect = const_cast<GraphComponent*>(this)->getClientSize();

    glX = (fx - minTick) * (2.0 / (maxTick - minTick)) - 1.0;
    glY = (fy - (minValue + maxValue)/2.0) * (2.0 / (maxValue - minValue));

    ix = (glX + 1.0) / (2.0 / (float)clientRect.getRight());
    iy = (glY - 1.0) / (-2.0 / (float)clientRect.getBottom());
  }

  void GraphComponent::onCreate() 
  {
    AztecGLFontCanvas::onCreate();

/*
    SCROLLINFO scrollInfo;

    scrollInfo.cbSize = sizeof(SCROLLINFO);
    scrollInfo.fMask = SIF_ALL;

    GetScrollInfo(SB_HORZ, &scrollInfo);

    scrollInfo.nMin = 0;
    scrollInfo.nMax = 100;
    scrollInfo.nPos = 50;

    SetScrollInfo(SB_HORZ, &scrollInfo);


    scrollInfo.cbSize = sizeof(SCROLLINFO);
    scrollInfo.fMask = SIF_ALL;

    GetScrollInfo(SB_VERT, &scrollInfo);

    scrollInfo.nMin = 0;
    scrollInfo.nMax = 100;
    scrollInfo.nPos = 50;

    SetScrollInfo(SB_VERT, &scrollInfo);
  

    return 0;*/
  }
}
