/*
token.c file of the Quest Source Code

Copyright 1997, 1998 Alexander Malmberg
Copyright 1996, Trey Harrison and Chris Carollo

This program is distributed under the GNU General Public License.
See legal.txt for more information.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>

#include "types.h"

#include "token.h"

#include "error.h"
#include "memory.h"

static char *filebuf=NULL;
static char *curpos;

static int global_flags;
static int flags;

static void(*PComment)(char *buf)=NULL;

char token[16384];
int linenum;

int LoadFile(char *filename,int fileflags,void (*parsecomment)(char *buf))
{
   FILE *f;
   int size;

   DoneFile();

   f=fopen(filename,"rb");
   if (!f)
      return 0;

   fseek(f,0,SEEK_END);
   size=ftell(f);
   fseek(f,0,SEEK_SET);

   filebuf=Q_malloc(size+1);
   if (!filebuf)
      return 0;

   fread(filebuf,1,size,f);

   filebuf[size]=0;

   fclose(f);

   PComment=parsecomment;
   global_flags=fileflags;
   curpos=filebuf;
   linenum=1;

   return 1;
}

void DoneFile(void)
{
   if (filebuf)
   {
      Q_free(filebuf);
      filebuf=NULL;
   }
}


static int SkipWhitespace(void)
{
   while ((*curpos<=32) && *curpos)
   {
      if (*curpos=='\n')
         linenum++;
      curpos++;
   }

   if (!*curpos)
      return 0;

   return 1;
}

static char *punctation[]=
{
"+",
"-",

"=",

"[",
"]",

"{",
"}",

"(",
")",

":",

"@"
};
#define NUM_PUNCTATION (sizeof(punctation)/sizeof(punctation[0]))

static int SkipToToken(int comments)
{
   char *t;

   do
   {
      if (!*curpos)
         return 0;

      if (!SkipWhitespace())
         return 0;

      if (flags&T_C)
      {
         if ((curpos[0]=='/') && (curpos[1]=='/'))
         {
            t=token;
            while (*curpos && (*curpos!='\n'))
               *t++=*curpos++;
            *t=0;
   
            if (!*curpos)
               return 0;
   
            if (PComment && comments)
            {
               PComment(token);
            }
   
            curpos++;
            linenum++;
            continue;
         }
   
         if ((curpos[0]=='/') && (curpos[1]=='*'))
         {
            curpos+=2;
            while (*curpos)
            {
               if (*curpos=='\n')
                  linenum++;
               if ((curpos[0]=='*') && (curpos[1]=='/'))
                  break;
               curpos++;
            }
            curpos+=2;
            if (!curpos)
               return 0;
            continue;
         }
      }

      if (flags&T_CFG)
      {
         if (*curpos=='#')
         {
            while (*curpos && (*curpos!='\n'))
               curpos++;
   
            if (!*curpos)
               return 0;
   
            curpos++;
            linenum++;
            continue;
         }
      }

      break;
   } while (*curpos);

   if (!*curpos)
      return 0;

   return 1;
}

static int IsName(int ch)
{
   if (isalnum(ch)) return 1;
   switch (ch)
   {
   case '_':
      return 1;
   default:
      return 0;
   }
}

int GetToken(int crossline,int tflags)
{
   int oldline;
   char *t;
   int i;

   if (!filebuf)
      Abort("GetToken","Internal error! GetToken() before LoadFile()!");

   flags=global_flags&tflags;

   oldline=linenum;

   if (!SkipToToken(1))
      return 0;

   if ((linenum!=oldline) && !crossline)
      return 0;

   if (!*curpos)
      return 0;

   t=token;

   if ((*curpos=='\"') && (flags&T_STRING))
   {
      *t++=*curpos++;
      while ((*curpos!='\"') && (*curpos))
      {
         if (*curpos=='\n')
            return 0;

         *t++=*curpos++;
      }

      if (!*curpos)
         return 0;

      *t++=*curpos++;
      *t=0;

      return 1;
   }

   if (flags&T_NUMBER)
   {
      if (((*curpos>='0') && (*curpos<='9')) || (*curpos=='-'))
      {
         *t++=*curpos++;
         while (((*curpos>='0') && (*curpos<='9')) || *curpos=='.')
            *t++=*curpos++;
   
         *t=0;
   
         return 1;
      }
   }

   if (flags&T_MISC)
   {
      for (i=0;i<NUM_PUNCTATION;i++)
      {
         if (!strncmp(curpos,punctation[i],strlen(punctation[i])))
         {
            strcpy(token,punctation[i]);
            curpos+=strlen(punctation[i]);
   
            return 1;
         }
      }
   }

   if ((flags&T_NAME) && IsName(*curpos))
   {
      while (IsName(*curpos))
         *t++=*curpos++;

      *t=0;

      return 1;
   }

   if (flags&T_ALLNAME)
   {
      while (*curpos)
      {
         if (*curpos<=32)
            break;

         *t++=*curpos++;
      }

      *t=0;

      return 1;
   }

   return 0;
}

int TokenAvailable(int crossline)
{
   int oldline,newline;
   char *oldpos;

   int i;

   oldline=linenum;
   oldpos=curpos;

   i=SkipToToken(0);

   newline=linenum;

   linenum=oldline;
   curpos=oldpos;

   if (!i)
      return 0;

   if ((newline!=oldline) && !crossline)
      return 0;

   return 1;
}

