/*
Copyright (C) 1996-1997 GX Media, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

/*
 * entity.cpp
 */

#include "stdafx.h"

#include "LCommon.h"
#include "Entity.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//==================== Entity ====================

#include "LCommon.h"

Entity::Entity(const wxString &cName) {

	int i;

	className = cName;

	// delWFGeom = false;
	wfGeomPtr = NULL;

	EntList *entList = EntList::Get();
	ASSERT(entList != NULL);

	entInfo = entList->FindEnt(className);
	if(entInfo) {
		wfGeomPtr = entInfo->Use(isModel, isAdjust);
	}
	else {
		if(className == wxT("cross1")) {
			wfGeomPtr = entList->cross1Ptr;
			isModel = false;
		}
		else if(className == wxT("cross2")) {
			wfGeomPtr = entList->cross2Ptr;
			isModel = false;
		}
		else {
			isModel = true;
			isAdjust = false;
		}
	}

	if(!isModel && !wfGeomPtr)
		wfGeomPtr = entList->undefPtr;

	if(wfGeomPtr)
		wfGeomPtr->Uses(1);

	for(i = 0; i < ENTITY_FLAGS; i++) {
		ea_flag[i] = false;
		notuser[i] = false;
	}

	notuser[EA_ORIGIN] = true;

	Check();
}

Entity::Entity(const Entity &src) {
	int i;

	wfGeomPtr = NULL;
	
	*this = src;
}

Entity::~Entity(void) {
	int i;

	if(entInfo)
		entInfo->DoneUse();
}

Entity &
Entity::operator=(const Entity &src) {
	ASSERT(!src.className.empty());

	int i;

	if (&src == this)
		return *this;

	className = src.className;

	if (src.wfGeomPtr != NULL)
		src.wfGeomPtr->Uses(1);
	if (wfGeomPtr != NULL)
		wfGeomPtr->Uses(-1);
	wfGeomPtr = src.wfGeomPtr;

	origin = src.origin;
	angle = src.angle;
	entInfo = src.entInfo;

	if ((ea_flag[EA_ORIGIN] = src.ea_flag[EA_ORIGIN]) != 0)
		origin = src.origin;
	if ((ea_flag[EA_ANGLE] = src.ea_flag[EA_ANGLE]) != 0)
		angle = src.angle;
	if ((ea_flag[EA_MANGLE] = src.ea_flag[EA_MANGLE]) != 0)
		mangle = src.mangle;

	for(i = 0; i < ENTITY_FLAGS; i++)
		notuser[i] = src.notuser[i];
  
	isModel = src.isModel;
  
	keyValues.reserve(src.keyValues.size());
	for(i = 0; i < src.keyValues.size(); i++) {
		keyValues.push_back(src.keyValues[i]);
	}

	return *this;
}

//===== Read / Write Entities. =====

Entity *
Entity::ReadEntity(LFile *inFile, int entNum,
                   int (*BrushCallBackFunc)(LFile *inFile),
				   int *pDone, int *pTotal,
				   bool (*ProgressFunc)(int percent)) {
	// PreCondition
	ASSERT(entNum >= -1);

	Entity *rtnVal = NULL;
	char *argBuf[MAX_ENTITY_KEYS];
	int i, argNum;
	char srch[20];

	for(i = 0; i < MAX_ENTITY_KEYS; i++)
		argBuf[i] = NULL;

	// Search for entity.
	if (entNum == -1)
		sprintf(srch, "{");
	else
		sprintf(srch, "{ // Entity %d", entNum);

	if (inFile->Search(srch) == NULL)
		goto ERROR_READ_ENTITY;

	if(ProgressFunc)
		ProgressFunc(++*pDone * 100 / *pTotal);

	argNum = 0;
	// Parse until end of entity Section.
	while (1) {
		if (inFile->GetNextLine() == NULL)
			goto ERROR_READ_ENTITY;
		else if (strncmp(inFile->GetLine(), "{", 1) == 0) {
			if(ProgressFunc)
				ProgressFunc(++*pDone * 100 / *pTotal);
			if (entNum != -1)
				goto ERROR_READ_ENTITY;
			if (BrushCallBackFunc(inFile) == -1)
				goto ERROR_READ_ENTITY;
		}
		else if (strncmp(inFile->GetLine(), "}", 1) == 0)
			break;
		else if (strncmp(inFile->GetLine(), "//", 2) == 0)
			continue;
		else if (strncmp(inFile->GetLine(), "\"", 1) == 0) {
			if (argNum >= MAX_ENTITY_KEYS)
				goto ERROR_READ_ENTITY;
			
			if(argBuf[argNum])
				delete argBuf[argNum];
			argBuf[argNum] = new char[strlen(inFile->GetLine()) + 1];
			strcpy(argBuf[argNum], inFile->GetLine());
			argNum++;
		}
	}

	if (argNum == 0)
		goto ERROR_READ_ENTITY;

	rtnVal = NewEntity((const char **) argBuf, argNum);
	if (rtnVal == NULL)
		goto ERROR_READ_ENTITY;

	goto RTN_READ_ENTITY;

ERROR_READ_ENTITY:
	if (entNum != -1)
		LError("Error load entity\n");

	if (rtnVal != NULL)
		delete rtnVal;
	rtnVal = NULL;

RTN_READ_ENTITY:
	for(i = 0; i < MAX_ENTITY_KEYS; i++)
		if(argBuf[i])
			delete argBuf[i];

	if(rtnVal)
		rtnVal->Check();

	return rtnVal;
}

Vector3d
Entity::GetVecPos(const Entity &entity) {
#if 1
	return entity.origin;
#else
	Vector3d minVec, maxVec;
	entity.wfGeomPtr->GetMinMax(minVec, maxVec);
	Vector3d vec = entity.origin;
	vec.SetX(vec.GetX() - minVec.GetX());
	vec.SetY(vec.GetY() - -minVec.GetY());
	vec.SetZ(vec.GetZ() - minVec.GetZ());
	return vec;
#endif
}

void
Entity::WriteEntity(FILE *outFile, const Entity &entity, int entNum) {
	// PreCondition.
	ASSERT(entNum >= -1);
	ASSERT(!entity.className.empty());

	int i;
	const Vector3d vec;
	char wkey[80], warg[512];

	if(entNum != -1)
		fprintf(outFile, "{ // Entity %d\n", entNum);

	strcpy(wkey, "\"classname\"");
	sprintf(warg, "\"%s\"", (const char*)entity.className.utf8_str());
	fprintf(outFile, "  %-16s%s\n", wkey, warg);

	// HACK: make sure to export origin, angle and mangle.
	if(entity.ea_flag[EA_ORIGIN])
	{
		Vector3d vec = Entity::GetVecPos(entity);
		strcpy(wkey, "\"origin\"");
		sprintf(warg, "\"%d %d %d\"", (int)ROUND(vec.GetX()),
				(int)ROUND(vec.GetY()), (int)ROUND(vec.GetZ()));
		fprintf(outFile, "  %-16s%s\n", wkey, warg);
	}

	if(entity.ea_flag[EA_ANGLE])
	{
		strcpy(wkey, "\"angle\"");
		sprintf(warg, "\"%d\"", entity.angle);
		fprintf(outFile, "  %-16s%s\n", wkey, warg);
	}

	if(entity.ea_flag[EA_MANGLE])
	{
		strcpy(wkey, "\"mangle\"");
		sprintf(warg, "\"%d %d %d\"", (int)ROUND(entity.mangle.GetX()),
			   (int)ROUND(entity.mangle.GetY()), (int)ROUND(entity.mangle.GetZ()));
		fprintf(outFile, "  %-16s%s\n", wkey, warg);
	}

	size_t numKeys = entity.keyValues.size();
	for(i = 0; i < numKeys; i++) {
		const KeyValue &kv = entity.keyValues[i];
		if(kv.key == wxT("origin") || kv.key == wxT("angle") ||
		   kv.key == wxT("mangle"))
		{
			// These are automatically generated.
			continue;
		}

		sprintf(wkey, "\"%s\"", (const char*)kv.key.utf8_str());
		sprintf(warg, "\"%s\"", (const char*)kv.value.utf8_str());

		if(strcmp(warg, "\"\""))
			fprintf(outFile, "  %-16s%s\n", wkey, warg);
	}
  
	if(entNum != -1)
		fprintf(outFile, "} // Entity %d\n", entNum);
}

Entity *Entity::ReadEntity(wxXmlNode *node, int entNum)
{
	ASSERT(entNum >= -1);

	// Read the class name.
	wxString className = node->GetAttribute(wxT("className"), wxT("(Null)"));

	// Create the entity.
	Entity *ent = new Entity(className);

	// Read the number of properties.
	long numprops = 0;
	node->GetAttribute(wxT("numprops"), wxT("0")).ToLong(&numprops);
	if(numprops < 0)
		numprops = 0;

	// Read the properties.
	ent->keyValues.reserve(numprops);
	wxXmlNode *propNode = node->GetChildren();
	for(long i = 0; i < numprops && propNode; propNode = propNode->GetNext())
	{
		// Ignore nodes that aren't properties.
		if(propNode->GetName() != wxT("prop"))
			continue;

		// Read the key.
		wxString key = propNode->GetAttribute(wxT("name"), wxT(""));

		// Ignore properties without a key.
		if(key.empty())
			continue;

		if(key == wxT("classname") || key == wxT("origin")
		  || key == wxT("mangle"))
			continue;

		// Read the value.
		wxString value = propNode->GetAttribute(wxT("value"), wxT(""));

		// Store the property.
		ent->SetKey(key, value);

		// Increase the key count.
		i++;
	}

	ent->Check();
	return ent;
}

bool Entity::WriteEntity(wxXmlNode *parent,
                 const Entity &entity, int entNum)
{
	ASSERT(entNum >= -1);
	ASSERT(!entity.className.empty());

	// Create the entity node.
	wxXmlNode *node = new wxXmlNode(wxXML_ELEMENT_NODE, wxT("entity"));
	parent->AddChild(node);

	// Store the class name.
	node->AddAttribute(wxT("className"), entity.className);

	// Store the entity id.
	node->AddAttribute(wxT("id"), wxString::Format(wxT("%d"), entNum));

	// Store each one of the key values.
	size_t numkv = entity.keyValues.size();
	node->AddAttribute(wxT("numprops"), wxString::Format(wxT("%d"), int(numkv)));
	for(size_t i = 0; i < numkv; i++)
	{
		const KeyValue &kv = entity.keyValues[i];

		// Create the key value node.
		wxXmlNode *kvNode = new wxXmlNode(wxXML_ELEMENT_NODE, wxT("prop"));
		node->AddChild(kvNode);

		const wxString &key = kv.key;
		wxString value;

		// Check for special keys.
		if(kv.key == wxT("origin") || kv.key == wxT("mangle"))
		{
			// The origin an mangle is given by the parent object.
			continue;
		}
		else if(kv.key == wxT("angle") && entity.ea_flag[EA_ANGLE])
		{
			value = wxString::Format(wxT("%d"), entity.GetAngle());
		}
		else
		{
			value = kv.value;
		}

		// Store the key and value.
		kvNode->AddAttribute(wxT("name"), key);
		kvNode->AddAttribute(wxT("value"), value);

	}

	return true;
}

//===== Key Manip =====

void
Entity::SetKey(const wxString &skey, const wxString &sarg) {
	size_t i;

	if(skey == wxT("origin"))
		ea_flag[EA_ORIGIN] = true;
	if(skey == wxT("angle"))
		SetAngle(atoi(sarg.utf8_str()));
	if(skey == wxT("mangle"))
		ea_flag[EA_MANGLE] = true;
      
	for(i = 0; i < keyValues.size(); i++) {
		KeyValue &kv = keyValues[i];
		if(kv.key == skey) {
			if(skey == wxT("classname") || skey == wxT("origin")) {
				ASSERT(0);
				LError("Entity::SetKey: not allowed to set %s that way", (const char*)skey.utf8_str());
				return;
			}

			kv.value = sarg;
			return;
		}
	}

	KeyValue kv;
	kv.key = skey;
	kv.value = sarg;
	keyValues.push_back(kv);
}

void
Entity::SetKeyNotUser(const wxString &skey, bool set) {
	for(size_t i = 0; i < keyValues.size(); i++)
	{
		KeyValue &kv = keyValues[i];
		if(kv.key == skey)
		{
			if(skey == wxT("angle")) {
				notuser[EA_ANGLE] = set;
				break;
			}
			if(skey == wxT("mangle")) {
				notuser[EA_MANGLE] = set;
				break;
			}
		}
	}
}

const wxString &
Entity::GetKey(const wxString &skey) const {
	static wxString empty;
	size_t i;
	for(i = 0; i < keyValues.size(); i++)
	{
		const KeyValue &kv = keyValues[i];
		if(kv.key == skey)
			return kv.value;
	}
	return empty;
}

int
Entity::GetNumKeys(void) {
	return keyValues.size();
}

const wxString &
Entity::GetKeyNum(int num) const {
	ASSERT(num >= 0 && num < keyValues.size());
	return keyValues[num].key;
}

void
Entity::DelKey(const wxString &skey) {
	KeyValues::iterator it = keyValues.begin();
	while(it != keyValues.end())
	{
		const KeyValue &kv = *it;
		if(kv.key == skey)
			keyValues.erase(it++);
		else
			it++;
	}

	if(skey == wxT("origin"))
		ea_flag[EA_ORIGIN] = false;
	if(skey == wxT("angle"))
		ea_flag[EA_ANGLE] = false;
	if(skey == wxT("mangle"))
		ea_flag[EA_MANGLE] = false;
}

//===== Entity Info =====

bool
Entity::IsWorldSpawn(void) const {
	return className == wxT("worldspawn");
}

bool
Entity::IsItemEntity(void) const {
	return !isModel;
}

bool
Entity::IsModelEntity(void) const {
	return isModel;
}

const wxString &
Entity::GetClassName(void) const {
	// Sanity Check.
	ASSERT(!className.empty());
	return className;
};

const Vector3d &
Entity::GetOrigin(void) const {
	return origin;
}

const int
Entity::GetAngle(void) const {
	return angle;
}

const Vector3d &
Entity::GetMAngle(void) const {
  return mangle;
}

void
Entity::SetOrigin(const Vector3d &pos) {
	ea_flag[EA_ORIGIN] = true;
	origin = pos;
}

void
Entity::SetAngle(int ang) {
	ea_flag[EA_ANGLE] = true;
	angle = ang;
}

void
Entity::SetMAngle(const Vector3d &pos) {
	ea_flag[EA_MANGLE] = true;
	mangle = pos;
}

bool
Entity::ParseKeys(const char *line, char **key, char **key2) {
	static char rtnVal1[80];
	static char rtnVal2[512];

	const char *c, *c1;
	int len;

	*key = *key2 = NULL;

	if ((c = strchr(line, '"')) == NULL)
		return false;
	if ((c1 = strchr(c + 1, '"')) == NULL)
		return false;
	len = Min(c1 - c - 1, 79);
	strncpy(rtnVal1, c + 1, len);
	rtnVal1[len] = '\0';
	*key = rtnVal1;

	if ((c = strchr(c1 + 1, '"')) == NULL)
		return true;
	if ((c1 = strchr(c + 1, '"')) == NULL)
		return true;
	len = Min(c1 - c - 1, 511);
	strncpy(rtnVal2, c + 1, len);
	rtnVal2[len] = '\0';
	*key2 = rtnVal2;

	return true;
}

Entity *
Entity::NewEntity(const char *keys[], int numKeys) {
	// Sanity Check.
	ASSERT(keys != NULL);

	int c, i, val;
	int cIndex;
	float x, y, z;
	Vector3d tmpVec;
	char *key, *arg;
	Entity *rtnVal = NULL;

	// Find the class name first.
	for(cIndex = 0; cIndex < numKeys; cIndex++) {
		if (ParseKeys(keys[cIndex], &key, &arg) && strcmp(key, "classname") == 0)
			break;
	}

	if (cIndex == numKeys || arg == NULL)
	{
		return NULL;
	}

	rtnVal = new Entity(wxString(arg, wxConvUTF8));

	for(i = 0; i < numKeys; i++) {
		if(!rtnVal->ParseKeys(keys[i], &key, &arg))
		{
			delete rtnVal;
			return NULL;
		}

		if(strcmpi(key, "classname") == 0) {
			continue;
		}
		else if(strcmpi(key, "origin") == 0) {
			if(sscanf(arg, "%f %f %f", &x, &y, &z) == 3) {
				tmpVec.NewVector(x, y, z);
				rtnVal->SetOrigin(tmpVec);
			}
		}
		else if(strcmpi(key, "angle") == 0) {
			if(sscanf(arg, "%d", &val) == 1)
				rtnVal->SetAngle(val);
		}

		KeyValue kv;
		kv.key = wxString(key, wxConvUTF8);
		kv.value = wxString(arg, wxConvUTF8);
		rtnVal->keyValues.push_back(kv);
	}
  
	rtnVal->Check();

	if(!rtnVal->className)
	{
		delete rtnVal;
		rtnVal = NULL;
	}

	return rtnVal;
}

void
Entity::Check(void) {
	int i;
	QEntEntry *entry;

	if(!entInfo)
		return;

	for(i = 0; i < entInfo->GetNumEntries(); i++) {
		entry = entInfo->GetEntryNum(i);
		const wxString &value = GetKey(entry->name);
		if(value.empty() && (!entry->def.empty() || entry->vartype == VAR_NOTUSER))
		{
			if(entry->name.CmpNoCase(wxT("origin")))
				SetKey(entry->name, entry->def);
		}
		if(entry->vartype == VAR_NOTUSER)
			SetKeyNotUser(entry->name);
	}

	int j;
	for(i = 0; i < GetNumKeys(); i++) {
		bool found = false;
		const wxString &key = GetKeyNum(i);

		for(j = 0; j < entInfo->GetNumEntries(); j++) {
			if(key == entInfo->GetEntryNum(j)->name) {
				found = true;
				break;
			}
		}

		if(!found)
			entInfo->AddUserEntry(key);
	}
}

bool
Entity::IsAttribSet(int attribID) const {
	return ea_flag[attribID];
}

void
Entity::UnSetAttrib(int attribID) {
	ea_flag[attribID] = false;
}

bool
Entity::IsNotUserAttrib(int attribID) const {
	return notuser[attribID];
}

bool
Entity::Need2AdjPos(void) const {
	return isAdjust;
}


