#include <string.h>
#include <stdlib.h>
#include <dirent.h>
#include <stdio.h>
#ifdef _WIN32
#include <windows.h>
#endif

#define MAX_LINE 1024

void map_surf_value(int value, char *result) {
    result[0] = '\0';  // Clear the result string
    if (value & 0x1) strcat(result, "Light&");
    if (value & 0x2) strcat(result, "Slick&");
    if (value & 0x4) strcat(result, "Sky&");
    if (value & 0x8) strcat(result, "Wrap&");
    if (value & 0x10) strcat(result, "Trans33&");
    if (value & 0x20) strcat(result, "Trans66&");
    if (value & 0x40) strcat(result, "Flowing&");
    if (value & 0x80) strcat(result, "NoDraw&");
    if (value & 0x100) strcat(result, "Hint&");
    if (value & 0x200) strcat(result, "Skip&");
    if (value & 0x400) strcat(result, "Specular&");
    if (value & 0x800) strcat(result, "Diffuse&");
    if (value & 0x1000) strcat(result, "Alpha&");
    if (value & 0x2000) strcat(result, "Mirror&");
    if (value & 0x4000) strcat(result, "Wndw33&");
    if (value & 0x8000) strcat(result, "Wndw66&");
    if (value & 0x10000) strcat(result, "Surf_0x10000&");
    if (value & 0x20000) strcat(result, "Surf_0x20000&");
    if (value & 0x40000) strcat(result, "Surf_0x40000&");
    if (value & 0x80000) strcat(result, "Water&");
    if (value & 0x100000) strcat(result, "Concrete&");
    if (value & 0x200000) strcat(result, "Fabric&");
    if (value & 0x400000) strcat(result, "Gravel&");
    if (value & 0x800000) strcat(result, "Metal&");
    if (value & 0x1000000) strcat(result, "Metal Lite&");
    if (value & 0x2000000) strcat(result, "Tin&");
    if (value & 0x4000000) strcat(result, "Tile&");
    if (value & 0x8000000) strcat(result, "Wood&");
    if (value & 0x10000000) strcat(result, "Reflect Fake&");
    if (value & 0x20000000) strcat(result, "Reflect Light&");
    if (value & 0x40000000) strcat(result, "Surf_0x40000000&");
    if (value & 0x80000000) strcat(result, "Surf_0x80000000&");

    // If no flags were set, output "0"
    if (result[0] == '\0') {
        strcpy(result, "0");
    } else {
        // Remove the trailing '&' if present
        size_t len = strlen(result);
        if (result[len - 1] == '&') {
            result[len - 1] = '\0';
        }
    }
}

void map_content_value(int value, char *result) {
    result[0] = '\0';  // Clear the result string
    if (value & 0x1) strcat(result, "Solid&");
    if (value & 0x2) strcat(result, "Window&");
    if (value & 0x4) strcat(result, "Aux&");
    if (value & 0x8) strcat(result, "Lava&");
    if (value & 0x10) strcat(result, "Slime&");
    if (value & 0x20) strcat(result, "Water&");
    if (value & 0x40) strcat(result, "Mist&");
    if (value & 0x80) strcat(result, "Fence&");
    if (value & 0x100) strcat(result, "Cont_0x100&");
    if (value & 0x200) strcat(result, "Cont_0x200&");
    if (value & 0x400) strcat(result, "Cont_0x400&");
    if (value & 0x800) strcat(result, "Cont_0x800&");
    if (value & 0x1000) strcat(result, "Cont_0x1000&");
    if (value & 0x2000) strcat(result, "Cont_0x2000&");
    if (value & 0x4000) strcat(result, "Cont_0x4000&");
    if (value & 0x8000) strcat(result, "Cont_0x8000&");
    if (value & 0x10000) strcat(result, "Player Clip&");
    if (value & 0x20000) strcat(result, "Monster Clip&");
    if (value & 0x40000) strcat(result, "current_0&");
    if (value & 0x80000) strcat(result, "current_90&");
    if (value & 0x100000) strcat(result, "current_180&");
    if (value & 0x200000) strcat(result, "current_270&");
    if (value & 0x400000) strcat(result, "current_up&");
    if (value & 0x800000) strcat(result, "current_down&");
    if (value & 0x1000000) strcat(result, "origin&");
    if (value & 0x2000000) strcat(result, "monster&");
    if (value & 0x4000000) strcat(result, "corpse&");
    if (value & 0x8000000) strcat(result, "detail&");
    if (value & 0x10000000) strcat(result, "translucent&");
    if (value & 0x20000000) strcat(result, "Ladder&");
    if (value & 0x40000000) strcat(result, "Cont_0x40000000&");
    if (value & 0x80000000) strcat(result, "Cont_0x80000000&");

    // If no flags were set, output "0"
    if (result[0] == '\0') {
        strcpy(result, "0");
    } else {
        // Remove the trailing '&' if present
        size_t len = strlen(result);
        if (result[len - 1] == '&') {
            result[len - 1] = '\0';
        }
    }
}

int starts_with_brush(const char *line) {
    // Skip any leading whitespace.
    while (isspace((unsigned char)*line)) {
        line++;
    }

    // Check for different brush header variants in a case-insensitive way.
    if (strncasecmp(line, "// brush", 8) == 0)
        return 1;
    if (strncasecmp(line, "// Brush", 8) == 0)
        return 1;
    if (strncasecmp(line, "{ //brush", 9) == 0)
        return 1;
    if (strncasecmp(line, "; Brush", 7) == 0)
        return 1;

    return 0;
}

int extract_texture_data(const char *line, char *output, const char *classname, int brush_id, int full_output, int hex_output) {
    char texture[MAX_LINE];
    float coords[9];       // The three sets of coordinates (9 floats)
    float adjustments[5];  // h_shift, v_shift, rotate, h_stretch, v_stretch
    int surf, content, value;
    char surf_result[MAX_LINE], content_result[MAX_LINE];

    int parsed = sscanf(line,
        " ( %f %f %f ) ( %f %f %f ) ( %f %f %f ) %s %f %f %f %f %f %d %d %d",
        &coords[0], &coords[1], &coords[2],
        &coords[3], &coords[4], &coords[5],
        &coords[6], &coords[7], &coords[8],
        texture,
        &adjustments[0], &adjustments[1], &adjustments[2],
        &adjustments[3], &adjustments[4],
        &content, &surf, &value);


    if (parsed == 18) {
        if (hex_output && full_output) {
            sprintf(output, "%s,%d,%s,%d,%d,%d,%.6f,%.6f,0x%x,0x%x,%d",
                    classname, brush_id, texture,
                    (int)adjustments[0], (int)adjustments[1], (int)adjustments[2],
                    adjustments[3], adjustments[4],
                    surf, content, value);
        } else if (hex_output) {
            sprintf(output, "%s,%d,%s,0x%x,0x%x",
                    classname, brush_id, texture, surf, content);
        } else if (full_output) {
            map_surf_value(surf, surf_result);
            map_content_value(content, content_result);
            sprintf(output, "%s,%d,%s,%d,%d,%d,%.6f,%.6f,%s,%s,%d",
                    classname, brush_id, texture,
                    (int)adjustments[0], (int)adjustments[1], (int)adjustments[2],
                    adjustments[3], adjustments[4],
                    surf_result, content_result, value);
        } else {
            map_surf_value(surf, surf_result);
            map_content_value(content, content_result);
            sprintf(output, "%s,%d,%s,%s,%s",
                    classname, brush_id, texture, surf_result, content_result);
        }
        return 1;  // Success
    }
    //printf("Failed to parse full line. Only %d items parsed.\n", parsed);
    //printf("Parsed %d values from line: %s\n", parsed, line);
    return 0;  // Failure
}



void process_map_file(const char *input_filename, const char *output_filename, int full_output, int hex_output) {
    FILE *input_file = fopen(input_filename, "r");
    FILE *output_file = fopen(output_filename, "w");

    if (input_file == NULL) {
        perror("Error opening input file");
        return;
    }

    if (output_file == NULL) {
        perror("Error opening output file");
        fclose(input_file);
        return;
    }

    char line[MAX_LINE];
    char extracted_line[MAX_LINE];
    int inside_brush = 0;
    int brush_id = -1;
    char classname[MAX_LINE] = "worldspawn";  // Default classname

    // Write CSV header based on output flags.
    if (hex_output && full_output) {
        fprintf(output_file, "classname,brush,texture,h_shift,v_shift,rotate,h_stretch,v_stretch,surf_hex,content_hex,value\n");
    } else if (hex_output) {
        fprintf(output_file, "classname,brush,texture,surf_hex,content_hex\n");
    } else if (full_output) {
        fprintf(output_file, "classname,brush,texture,h_shift,v_shift,rotate,h_stretch,v_stretch,surf,content,value\n");
    } else {
        fprintf(output_file, "classname,brush,texture,surf,content\n");
    }

    while (fgets(line, sizeof(line), input_file)) {
        // Create a trimmed pointer (skipping any leading whitespace).
        char *trimmed = line;
        while (isspace((unsigned char)*trimmed)) {
            trimmed++;
        }

        // Skip comment lines unless they indicate a brush header or contain a classname.
        if (trimmed[0] == ';' || trimmed[0] == '/') {
            if (starts_with_brush(trimmed)) {
                // Determine brush id using recognized formats (case-insensitive).
                if (strncasecmp(trimmed, "// brush", 8) == 0 || strncasecmp(trimmed, "// Brush", 8) == 0) {
                    if (sscanf(trimmed, "// brush %d", &brush_id) != 1) {
                        sscanf(trimmed, "// Brush %d", &brush_id);
                    }
                    inside_brush = 1;
                    continue;
                } else if (strncasecmp(trimmed, "{ //brush", 9) == 0) {
                    sscanf(trimmed, "{ //brush %d", &brush_id);
                    inside_brush = 1;
                    continue;
                } else if (strncasecmp(trimmed, "; Brush", 7) == 0) {
                    sscanf(trimmed, "; Brush %d", &brush_id);
                    inside_brush = 1;
                    continue;
                }
            }
            // Also check for classname lines that might be commented.
            if (strstr(trimmed, "\"classname\"") != NULL) {
                sscanf(trimmed, "\"classname\" \"%[^\"]\"", classname);
            }
            continue;
        }

        // If the line contains a classname property outside comments.
        if (strstr(trimmed, "\"classname\"") != NULL) {
            sscanf(trimmed, "\"classname\" \"%[^\"]\"", classname);
        }
        // Detect brush header not caught by earlier comment block.
        else if (starts_with_brush(trimmed)) {
            if (strncasecmp(trimmed, "// brush", 8) == 0 || strncasecmp(trimmed, "// Brush", 8) == 0) {
                if (sscanf(trimmed, "// brush %d", &brush_id) != 1) {
                    sscanf(trimmed, "// Brush %d", &brush_id);
                }
            } else if (strncasecmp(trimmed, "{ //brush", 9) == 0) {
                sscanf(trimmed, "{ //brush %d", &brush_id);
            } else if (strncasecmp(trimmed, "; Brush", 7) == 0) {
                sscanf(trimmed, "; Brush %d", &brush_id);
            }
            inside_brush = 1;
            continue;
        }
        // Ignore opening brace lines inside a brush block.
        else if (inside_brush && trimmed[0] == '{') {
            continue;
        }
        // If we're ending a brush block.
        else if (inside_brush && trimmed[0] == '}') {
            inside_brush = 0;
            continue;
        }
        // Process geometry/texture data lines inside a brush block.
        else if (inside_brush) {
            // Pass the trimmed geometry line to your extraction function.
            if (extract_texture_data(trimmed, extracted_line, classname, brush_id, full_output, hex_output)) {
                fprintf(output_file, "%s\n", extracted_line);
            }
        }
    }

    fclose(input_file);
    fclose(output_file);
    printf("File processing complete. Output saved to: %s\n", output_filename);
}

int is_wildcard(const char *filename) {
    // Simple check for wildcard
    return strstr(filename, "*") != NULL;
}

int main(int argc, char *argv[]) {
    if (argc < 2 || argc > 4) {
        printf("Map2csv v1.0 by FREDZ\n");
        printf("Convert only brushes of map file to csv\n");
        printf("Does not work on 220 valve maps. Works best on KpRadiant maps.\n\n");
        printf("full: show everything from surface brushes, no coordinates\n");
        printf("hex:  show hex values instead of flags\n");
        printf("\nUsage: map2csv <filename or pattern> [full] [hex]\n");
        return EXIT_FAILURE;
    }

    const char *input_filename = argv[1];
    char output_filename[MAX_LINE];
    int full_output = 0;
    int hex_output = 0;

    for (int i = 2; i < argc; i++) {
        if (strcmp(argv[i], "full") == 0) {
            full_output = 1;
        } else if (strcmp(argv[i], "hex") == 0) {
            hex_output = 1;
        }
    }

    if (!is_wildcard(input_filename)) {
        // Single file mode
        printf("Processing: %s\n", input_filename);

        // Generate output filename
        strcpy(output_filename, input_filename);
        char *dot = strrchr(output_filename, '.');
        if (dot && strcmp(dot, ".map") == 0) {
            strcpy(dot, ".csv");
        } else {
            strcat(output_filename, ".csv");
        }

        process_map_file(input_filename, output_filename, full_output, hex_output);
    } else {
        // Wildcard mode (process all matching files)
#ifdef _WIN32
        WIN32_FIND_DATA find_data;
        HANDLE hFind = FindFirstFile(input_filename, &find_data);

        if (hFind == INVALID_HANDLE_VALUE) {
            printf("No files matching pattern found.\n");
            return EXIT_FAILURE;
        }

        do {
            if (!(find_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
                printf("Processing: %s\n", find_data.cFileName);

                // Generate output filename
                strcpy(output_filename, find_data.cFileName);
                char *dot = strrchr(output_filename, '.');
                if (dot && strcmp(dot, ".map") == 0) {
                    strcpy(dot, ".csv");
                } else {
                    strcat(output_filename, ".csv");
                }

                process_map_file(find_data.cFileName, output_filename, full_output, hex_output);
            }
        } while (FindNextFile(hFind, &find_data) != 0);

        FindClose(hFind);
#else
        DIR *dir = opendir(".");
        if (!dir) {
            perror("Error opening directory");
            return EXIT_FAILURE;
        }

        struct dirent *entry;
        while ((entry = readdir(dir)) != NULL) {
            if (strstr(entry->d_name, ".map") != NULL && strstr(entry->d_name, ".map") == entry->d_name + strlen(entry->d_name) - 4) {
                printf("Processing: %s\n", entry->d_name);

                // Generate output filename
                strcpy(output_filename, entry->d_name);
                char *dot = strrchr(output_filename, '.');
                if (dot && strcmp(dot, ".map") == 0) {
                    strcpy(dot, ".csv");
                } else {
                    strcat(output_filename, ".csv");
                }

                process_map_file(entry->d_name, output_filename, full_output, hex_output);
            }
        }

        closedir(dir);
#endif
        printf("All files processed.\n");
    }

    return EXIT_SUCCESS;
}



